// SPDX-License-Identifier: GPL-2.0-only OR MIT
/*
 * Copyright (C) 2023 The Falco Authors.
 *
 * This file is dual licensed under either the MIT or GPL 2. See MIT.txt
 * or GPL2.txt for full copies of the license.
 */

#include <helpers/interfaces/fixed_size_event.h>
#include <helpers/interfaces/variable_size_event.h>

/*=============================== EXIT EVENT ===========================*/

SEC("tp_btf/sys_exit")
int BPF_PROG(pwrite64_x, struct pt_regs *regs, long ret) {
	struct auxiliary_map *auxmap = auxmap__get();
	if(!auxmap) {
		return 0;
	}

	auxmap__preload_event_header(auxmap, PPME_SYSCALL_PWRITE_X);

	/*=============================== COLLECT PARAMETERS  ===========================*/

	/* Parameter 1: res (type: PT_ERRNO) */
	auxmap__store_s64_param(auxmap, ret);

	/* If the syscall doesn't fail we use the return value as `size`
	 * otherwise we need to rely on the syscall parameter provided by the user.
	 */
	dynamic_snaplen_args snaplen_args = {
	        .only_port_range = false,
	        .evt_type = PPME_SYSCALL_PWRITE_X,
	};
	size_t size = extract__syscall_argument(regs, 2);
	int64_t bytes_to_read = ret > 0 ? ret : size;
	uint16_t snaplen = maps__get_snaplen();
	apply_dynamic_snaplen(regs, &snaplen, &snaplen_args);
	if((int64_t)snaplen > bytes_to_read) {
		snaplen = bytes_to_read;
	}

	/* Parameter 2: data (type: PT_BYTEBUF) */
	unsigned long data_pointer = extract__syscall_argument(regs, 1);
	auxmap__store_bytebuf_param(auxmap, data_pointer, snaplen, USER);

	/* Parameter 3: fd (type: PT_FD) */
	int32_t fd = (int32_t)extract__syscall_argument(regs, 0);
	auxmap__store_s64_param(auxmap, (int64_t)fd);

	/* Parameter 4: size (type: PT_UINT32) */
	auxmap__store_u32_param(auxmap, size);

	/* Parameter 5: pos (type: PT_UINT64) */
	uint64_t pos = (uint64_t)extract__syscall_argument(regs, 3);
	auxmap__store_u64_param(auxmap, pos);

	/*=============================== COLLECT PARAMETERS  ===========================*/

	auxmap__finalize_event_header(auxmap);

	auxmap__submit_event(auxmap);

	return 0;
}

/*=============================== EXIT EVENT ===========================*/
