// SPDX-License-Identifier: GPL-2.0-only OR MIT
/*

Copyright (C) 2023 The Falco Authors.

This file is dual licensed under either the MIT or GPL 2. See MIT.txt
or GPL2.txt for full copies of the license.

*/

/* Some rules on how to populate this table:
 * - Enter and exit events should have the same flags unless the exit one is `EC_UNKNOWN`.
 *
 * - The `ppm_event_category` is composed of 2 parts:
 *
 *    1. The highest bits represent the event category:
 *  	- `EC_SYSCALL`
 *   	- `EC_TRACEPOINT
 *   	- `EC_PLUGIN`
 *   	- `EC_METAEVENT`
 *
 *    2. The lowest bits represent the syscall category to which the specific event belongs.
 *
 *   All events must have only one syscall category and one event category. Exception: events
 *   marked with `EC_UNKNOWN` flag must only have the syscall category equal to `EC_UNKNOWN`.
 *
 * - All events that are no more sent by our drivers must have the `EF_OLD_VERSION` flag.
 *
 * - Events marked with `EC_UNKNOWN` must have a name equal to `NA`.
 *
 * - All events that have the "EF_USES_FD" flag should return as one of the parameters a file
 *descriptor. "libsinsp" will try to access the parameter and use it as a file descriptor. If the
 *event has 0 parameters but has the "EF_USES_FD" flag then a runtime error will occur shutting down
 *the process. Furthermore if an exit event has the "EF_USES_FD" then also the related enter event
 *must have it (following the logic described above). Otherwise the exit event will not trigger
 *"libsinsp" code in order to properly manage the file descriptor returned by the exit event.
 *
 * - The only kind of change permitted for pre-existent events is adding parameters. If you need to
 *modify or remove some existing parameters you must create a new event pair. The new enum name
 *should be equal to the previous one but with the version bumped by 1. Consider the
 *`PPME_SYSCALL_EXECVE_19_X` event as an example, if you want to create a new version for it, the
 *new enum will be called `PPME_SYSCALL_EXECVE_20_X`.
 *
 * - All the versions of the same event must have the same name
 */

#include "ppm_events_public.h"

#pragma GCC diagnostic push
#pragma GCC diagnostic ignored "-Wmissing-field-initializers"
const struct ppm_event_info g_event_info[] = {
        [PPME_GENERIC_E] = {"syscall",
                            EC_OTHER | EC_SYSCALL,
                            EF_OLD_VERSION | EF_CONVERTER_MANAGED,
                            2,
                            {{"ID", PT_SYSCALLID, PF_DEC}, {"nativeID", PT_UINT16, PF_DEC}}},
        [PPME_GENERIC_X] = {"syscall",
                            EC_OTHER | EC_SYSCALL,
                            EF_CONVERTER_MANAGED,
                            2,
                            {{"ID", PT_SYSCALLID, PF_DEC}, {"nativeID", PT_UINT16, PF_DEC}}},
        [PPME_SYSCALL_OPEN_E] = {"open",
                                 EC_FILE | EC_SYSCALL,
                                 EF_CREATES_FD | EF_MODIFIES_STATE | EF_CONVERTER_MANAGED,
                                 3,
                                 {{"name", PT_FSPATH, PF_NA},
                                  {"flags", PT_FLAGS32, PF_HEX, file_flags},
                                  {"mode", PT_UINT32, PF_OCT}}},
        [PPME_SYSCALL_OPEN_X] = {"open",
                                 EC_FILE | EC_SYSCALL,
                                 EF_CREATES_FD | EF_MODIFIES_STATE | EF_CONVERTER_MANAGED,
                                 6,
                                 {{"fd", PT_FD, PF_DEC},
                                  {"name", PT_FSPATH, PF_NA},
                                  {"flags", PT_FLAGS32, PF_HEX, file_flags},
                                  {"mode", PT_UINT32, PF_OCT},
                                  {"dev", PT_UINT32, PF_HEX},
                                  {"ino", PT_UINT64, PF_DEC}}},
        [PPME_SYSCALL_CLOSE_E] = {"close",
                                  EC_IO_OTHER | EC_SYSCALL,
                                  EF_OLD_VERSION | EF_DESTROYS_FD | EF_USES_FD | EF_MODIFIES_STATE |
                                          EF_CONVERTER_MANAGED,
                                  1,
                                  {{"fd", PT_FD, PF_DEC}}},
        [PPME_SYSCALL_CLOSE_X] = {"close",
                                  EC_IO_OTHER | EC_SYSCALL,
                                  EF_DESTROYS_FD | EF_USES_FD | EF_MODIFIES_STATE |
                                          EF_CONVERTER_MANAGED,
                                  2,
                                  {{"res", PT_ERRNO, PF_DEC}, {"fd", PT_FD, PF_DEC}}},
        [PPME_SYSCALL_READ_E] = {"read",
                                 EC_IO_READ | EC_SYSCALL,
                                 EF_OLD_VERSION | EF_USES_FD | EF_READS_FROM_FD |
                                         EF_CONVERTER_MANAGED,
                                 2,
                                 {{"fd", PT_FD, PF_DEC}, {"size", PT_UINT32, PF_DEC}}},
        [PPME_SYSCALL_READ_X] = {"read",
                                 EC_IO_READ | EC_SYSCALL,
                                 EF_USES_FD | EF_READS_FROM_FD | EF_CONVERTER_MANAGED,
                                 4,
                                 {{"res", PT_ERRNO, PF_DEC},
                                  {"data", PT_BYTEBUF, PF_NA},
                                  {"fd", PT_FD, PF_DEC},
                                  {"size", PT_UINT32, PF_DEC}}},
        [PPME_SYSCALL_WRITE_E] = {"write",
                                  EC_IO_WRITE | EC_SYSCALL,
                                  EF_OLD_VERSION | EF_USES_FD | EF_WRITES_TO_FD |
                                          EF_CONVERTER_MANAGED,
                                  2,
                                  {{"fd", PT_FD, PF_DEC}, {"size", PT_UINT32, PF_DEC}}},
        [PPME_SYSCALL_WRITE_X] = {"write",
                                  EC_IO_WRITE | EC_SYSCALL,
                                  EF_USES_FD | EF_WRITES_TO_FD | EF_CONVERTER_MANAGED,
                                  4,
                                  {{"res", PT_ERRNO, PF_DEC},
                                   {"data", PT_BYTEBUF, PF_NA},
                                   {"fd", PT_FD, PF_DEC},
                                   {"size", PT_UINT32, PF_DEC}}},
        [PPME_SYSCALL_BRK_1_E] = {"brk",
                                  EC_MEMORY | EC_SYSCALL,
                                  EF_OLD_VERSION | EF_CONVERTER_MANAGED,
                                  1,
                                  {{"size", PT_UINT32, PF_DEC}}},
        [PPME_SYSCALL_BRK_1_X] = {"brk",
                                  EC_MEMORY | EC_SYSCALL,
                                  EF_OLD_VERSION | EF_CONVERTER_MANAGED,
                                  1,
                                  {{"res", PT_UINT64, PF_HEX}}},
        [PPME_SYSCALL_EXECVE_8_E] = {"execve",
                                     EC_PROCESS | EC_SYSCALL,
                                     EF_OLD_VERSION | EF_MODIFIES_STATE | EF_CONVERTER_MANAGED,
                                     0},
        [PPME_SYSCALL_EXECVE_8_X] = {"execve",
                                     EC_PROCESS | EC_SYSCALL,
                                     EF_OLD_VERSION | EF_MODIFIES_STATE | EF_CONVERTER_MANAGED,
                                     8,
                                     {{"res", PT_ERRNO, PF_DEC},
                                      {"exe", PT_CHARBUF, PF_NA},
                                      {"args", PT_BYTEBUF, PF_NA},
                                      {"tid", PT_PID, PF_DEC},
                                      {"pid", PT_PID, PF_DEC},
                                      {"ptid", PT_PID, PF_DEC},
                                      {"cwd", PT_CHARBUF, PF_NA},
                                      {"fdlimit", PT_UINT64, PF_DEC}}},
        [PPME_SYSCALL_CLONE_11_E] = {"clone",
                                     EC_PROCESS | EC_SYSCALL,
                                     EF_OLD_VERSION | EF_MODIFIES_STATE,
                                     0},
        [PPME_SYSCALL_CLONE_11_X] = {"clone",
                                     EC_PROCESS | EC_SYSCALL,
                                     EF_OLD_VERSION | EF_MODIFIES_STATE | EF_CONVERTER_MANAGED,
                                     11,
                                     {{"res", PT_PID, PF_DEC},
                                      {"exe", PT_CHARBUF, PF_NA},
                                      {"args", PT_BYTEBUF, PF_NA},
                                      {"tid", PT_PID, PF_DEC},
                                      {"pid", PT_PID, PF_DEC},
                                      {"ptid", PT_PID, PF_DEC},
                                      {"cwd", PT_CHARBUF, PF_NA},
                                      {"fdlimit", PT_INT64, PF_DEC},
                                      {"flags", PT_FLAGS32, PF_HEX, clone_flags},
                                      {"uid", PT_UINT32, PF_DEC},
                                      {"gid", PT_UINT32, PF_DEC}}},
        [PPME_PROCEXIT_E] = {"procexit",
                             EC_PROCESS | EC_TRACEPOINT,
                             EF_OLD_VERSION | EF_MODIFIES_STATE | EF_CONVERTER_MANAGED,
                             0},
        [PPME_PROCEXIT_X] = {"NA", EC_UNKNOWN, EF_UNUSED, 0},
        [PPME_SOCKET_SOCKET_E] = {"socket",
                                  EC_NET | EC_SYSCALL,
                                  EF_OLD_VERSION | EF_CREATES_FD | EF_MODIFIES_STATE |
                                          EF_CONVERTER_MANAGED,
                                  3,
                                  {{"domain", PT_ENUMFLAGS32, PF_DEC, socket_families},
                                   {"type", PT_UINT32, PF_DEC},
                                   {"proto", PT_UINT32, PF_DEC}}},
        [PPME_SOCKET_SOCKET_X] = {"socket",
                                  EC_NET | EC_SYSCALL,
                                  EF_CREATES_FD | EF_MODIFIES_STATE | EF_CONVERTER_MANAGED,
                                  4,
                                  {{"fd", PT_FD, PF_DEC},
                                   {"domain", PT_ENUMFLAGS32, PF_DEC, socket_families},
                                   {"type", PT_UINT32, PF_DEC},
                                   {"proto", PT_UINT32, PF_DEC}}},
        [PPME_SOCKET_BIND_E] = {"bind",
                                EC_NET | EC_SYSCALL,
                                EF_OLD_VERSION | EF_USES_FD | EF_MODIFIES_STATE |
                                        EF_CONVERTER_MANAGED,
                                1,
                                {{"fd", PT_FD, PF_DEC}}},
        [PPME_SOCKET_BIND_X] = {"bind",
                                EC_NET | EC_SYSCALL,
                                EF_USES_FD | EF_MODIFIES_STATE | EF_CONVERTER_MANAGED,
                                3,
                                {{"res", PT_ERRNO, PF_DEC},
                                 {"addr", PT_SOCKADDR, PF_NA},
                                 {"fd", PT_FD, PF_DEC}}},
        [PPME_SOCKET_CONNECT_E] = {"connect",
                                   EC_NET | EC_SYSCALL,
                                   EF_USES_FD | EF_MODIFIES_STATE | EF_CONVERTER_MANAGED,
                                   2,
                                   {{"fd", PT_FD, PF_DEC}, {"addr", PT_SOCKADDR, PF_NA}}},
        [PPME_SOCKET_CONNECT_X] = {"connect",
                                   EC_NET | EC_SYSCALL,
                                   EF_USES_FD | EF_MODIFIES_STATE | EF_CONVERTER_MANAGED,
                                   4,
                                   {{"res", PT_ERRNO, PF_DEC},
                                    {"tuple", PT_SOCKTUPLE, PF_NA},
                                    {"fd", PT_FD, PF_DEC},
                                    {"addr", PT_SOCKADDR, PF_NA}}},
        [PPME_SOCKET_LISTEN_E] = {"listen",
                                  EC_NET | EC_SYSCALL,
                                  EF_OLD_VERSION | EF_USES_FD | EF_CONVERTER_MANAGED,
                                  2,
                                  {{"fd", PT_FD, PF_DEC}, {"backlog", PT_INT32, PF_DEC}}},
        [PPME_SOCKET_LISTEN_X] = {"listen",
                                  EC_NET | EC_SYSCALL,
                                  EF_USES_FD | EF_CONVERTER_MANAGED,
                                  3,
                                  {{"res", PT_ERRNO, PF_DEC},
                                   {"fd", PT_FD, PF_DEC},
                                   {"backlog", PT_INT32, PF_DEC}}},
        [PPME_SOCKET_ACCEPT_E] = {"accept",
                                  EC_NET | EC_SYSCALL,
                                  EF_OLD_VERSION | EF_CREATES_FD | EF_MODIFIES_STATE,
                                  0},
        [PPME_SOCKET_ACCEPT_X] = {"accept",
                                  EC_NET | EC_SYSCALL,
                                  EF_OLD_VERSION | EF_CREATES_FD | EF_MODIFIES_STATE |
                                          EF_CONVERTER_MANAGED,
                                  3,
                                  {{"fd", PT_FD, PF_DEC},
                                   {"tuple", PT_SOCKTUPLE, PF_NA},
                                   {"queuepct", PT_UINT8, PF_DEC}}},
        [PPME_SOCKET_SEND_E] = {"send",
                                EC_IO_WRITE | EC_SYSCALL,
                                EF_OLD_VERSION | EF_USES_FD | EF_WRITES_TO_FD |
                                        EF_CONVERTER_MANAGED,
                                2,
                                {{"fd", PT_FD, PF_DEC}, {"size", PT_UINT32, PF_DEC}}},
        [PPME_SOCKET_SEND_X] = {"send",
                                EC_IO_WRITE | EC_SYSCALL,
                                EF_USES_FD | EF_WRITES_TO_FD | EF_CONVERTER_MANAGED,
                                5,
                                {{"res", PT_ERRNO, PF_DEC},
                                 {"data", PT_BYTEBUF, PF_NA},
                                 {"fd", PT_FD, PF_DEC},
                                 {"size", PT_UINT32, PF_DEC},
                                 {"tuple", PT_SOCKTUPLE, PF_NA}}},
        [PPME_SOCKET_SENDTO_E] = {"sendto",
                                  EC_IO_WRITE | EC_SYSCALL,
                                  EF_OLD_VERSION | EF_USES_FD | EF_WRITES_TO_FD |
                                          EF_MODIFIES_STATE | EF_CONVERTER_MANAGED,
                                  3,
                                  {{"fd", PT_FD, PF_DEC},
                                   {"size", PT_UINT32, PF_DEC},
                                   {"tuple", PT_SOCKTUPLE, PF_NA}}},
        [PPME_SOCKET_SENDTO_X] = {"sendto",
                                  EC_IO_WRITE | EC_SYSCALL,
                                  EF_USES_FD | EF_WRITES_TO_FD | EF_MODIFIES_STATE |
                                          EF_CONVERTER_MANAGED,
                                  5,
                                  {{"res", PT_ERRNO, PF_DEC},
                                   {"data", PT_BYTEBUF, PF_NA},
                                   {"fd", PT_FD, PF_DEC},
                                   {"size", PT_UINT32, PF_DEC},
                                   {"tuple", PT_SOCKTUPLE, PF_NA}}},
        [PPME_SOCKET_RECV_E] = {"recv",
                                EC_IO_READ | EC_SYSCALL,
                                EF_OLD_VERSION | EF_USES_FD | EF_READS_FROM_FD |
                                        EF_CONVERTER_MANAGED,
                                2,
                                {{"fd", PT_FD, PF_DEC}, {"size", PT_UINT32, PF_DEC}}},
        [PPME_SOCKET_RECV_X] = {"recv",
                                EC_IO_READ | EC_SYSCALL,
                                EF_USES_FD | EF_READS_FROM_FD | EF_CONVERTER_MANAGED,
                                5,
                                {{"res", PT_ERRNO, PF_DEC},
                                 {"data", PT_BYTEBUF, PF_NA},
                                 {"fd", PT_FD, PF_DEC},
                                 {"size", PT_UINT32, PF_DEC},
                                 {"tuple", PT_SOCKTUPLE, PF_NA}}},
        [PPME_SOCKET_RECVFROM_E] = {"recvfrom",
                                    EC_IO_READ | EC_SYSCALL,
                                    EF_OLD_VERSION | EF_USES_FD | EF_READS_FROM_FD |
                                            EF_MODIFIES_STATE | EF_CONVERTER_MANAGED,
                                    2,
                                    {{"fd", PT_FD, PF_DEC}, {"size", PT_UINT32, PF_DEC}}},
        [PPME_SOCKET_RECVFROM_X] = {"recvfrom",
                                    EC_IO_READ | EC_SYSCALL,
                                    EF_USES_FD | EF_READS_FROM_FD | EF_MODIFIES_STATE |
                                            EF_CONVERTER_MANAGED,
                                    5,
                                    {{"res", PT_ERRNO, PF_DEC},
                                     {"data", PT_BYTEBUF, PF_NA},
                                     {"tuple", PT_SOCKTUPLE, PF_NA},
                                     {"fd", PT_FD, PF_DEC},
                                     {"size", PT_UINT32, PF_DEC}}},
        [PPME_SOCKET_SHUTDOWN_E] = {"shutdown",
                                    EC_NET | EC_SYSCALL,
                                    EF_OLD_VERSION | EF_USES_FD | EF_MODIFIES_STATE |
                                            EF_CONVERTER_MANAGED,
                                    2,
                                    {{"fd", PT_FD, PF_DEC},
                                     {"how", PT_ENUMFLAGS8, PF_HEX, shutdown_how}}},
        [PPME_SOCKET_SHUTDOWN_X] = {"shutdown",
                                    EC_NET | EC_SYSCALL,
                                    EF_USES_FD | EF_MODIFIES_STATE | EF_CONVERTER_MANAGED,
                                    3,
                                    {{"res", PT_ERRNO, PF_DEC},
                                     {"fd", PT_FD, PF_DEC},
                                     {"how", PT_ENUMFLAGS8, PF_HEX, shutdown_how}}},
        [PPME_SOCKET_GETSOCKNAME_E] = {"getsockname", EC_NET | EC_SYSCALL, EF_OLD_VERSION, 0},
        [PPME_SOCKET_GETSOCKNAME_X] = {"getsockname", EC_NET | EC_SYSCALL, EF_NONE, 0},
        [PPME_SOCKET_GETPEERNAME_E] = {"getpeername", EC_NET | EC_SYSCALL, EF_OLD_VERSION, 0},
        [PPME_SOCKET_GETPEERNAME_X] = {"getpeername", EC_NET | EC_SYSCALL, EF_NONE, 0},
        [PPME_SOCKET_SOCKETPAIR_E] = {"socketpair",
                                      EC_IPC | EC_SYSCALL,
                                      EF_OLD_VERSION | EF_CREATES_FD | EF_MODIFIES_STATE |
                                              EF_CONVERTER_MANAGED,
                                      3,
                                      {{"domain", PT_ENUMFLAGS32, PF_DEC, socket_families},
                                       {"type", PT_UINT32, PF_DEC},
                                       {"proto", PT_UINT32, PF_DEC}}},
        [PPME_SOCKET_SOCKETPAIR_X] = {"socketpair",
                                      EC_IPC | EC_SYSCALL,
                                      EF_CREATES_FD | EF_MODIFIES_STATE | EF_CONVERTER_MANAGED,
                                      8,
                                      {{"res", PT_ERRNO, PF_DEC},
                                       {"fd1", PT_FD, PF_DEC},
                                       {"fd2", PT_FD, PF_DEC},
                                       {"source", PT_UINT64, PF_HEX},
                                       {"peer", PT_UINT64, PF_HEX},
                                       {"domain", PT_ENUMFLAGS32, PF_DEC, socket_families},
                                       {"type", PT_UINT32, PF_DEC},
                                       {"proto", PT_UINT32, PF_DEC}}},
        [PPME_SOCKET_SETSOCKOPT_E] = {"setsockopt", EC_NET | EC_SYSCALL, EF_OLD_VERSION, 0},
        [PPME_SOCKET_SETSOCKOPT_X] =
                {"setsockopt",
                 EC_NET | EC_SYSCALL,
                 EF_USES_FD,
                 6,
                 {{"res", PT_ERRNO, PF_DEC},
                  {"fd", PT_FD, PF_DEC},
                  {"level", PT_ENUMFLAGS8, PF_DEC, sockopt_levels},
                  {"optname", PT_ENUMFLAGS8, PF_DEC, sockopt_options},
                  {"val", PT_DYN, PF_DEC, sockopt_dynamic_param, PPM_SOCKOPT_IDX_MAX},
                  {"optlen", PT_UINT32, PF_DEC}}},
        [PPME_SOCKET_GETSOCKOPT_E] = {"getsockopt",
                                      EC_NET | EC_SYSCALL,
                                      EF_OLD_VERSION | EF_MODIFIES_STATE,
                                      0},
        [PPME_SOCKET_GETSOCKOPT_X] =
                {"getsockopt",
                 EC_NET | EC_SYSCALL,
                 EF_USES_FD | EF_MODIFIES_STATE,
                 6,
                 {{"res", PT_ERRNO, PF_DEC},
                  {"fd", PT_FD, PF_DEC},
                  {"level", PT_ENUMFLAGS8, PF_DEC, sockopt_levels},
                  {"optname", PT_ENUMFLAGS8, PF_DEC, sockopt_options},
                  {"val", PT_DYN, PF_DEC, sockopt_dynamic_param, PPM_SOCKOPT_IDX_MAX},
                  {"optlen", PT_UINT32, PF_DEC}}},
        [PPME_SOCKET_SENDMSG_E] = {"sendmsg",
                                   EC_IO_WRITE | EC_SYSCALL,
                                   EF_OLD_VERSION | EF_USES_FD | EF_WRITES_TO_FD |
                                           EF_MODIFIES_STATE | EF_CONVERTER_MANAGED,
                                   3,
                                   {{"fd", PT_FD, PF_DEC},
                                    {"size", PT_UINT32, PF_DEC},
                                    {"tuple", PT_SOCKTUPLE, PF_NA}}},
        [PPME_SOCKET_SENDMSG_X] = {"sendmsg",
                                   EC_IO_WRITE | EC_SYSCALL,
                                   EF_USES_FD | EF_WRITES_TO_FD | EF_MODIFIES_STATE |
                                           EF_CONVERTER_MANAGED,
                                   5,
                                   {{"res", PT_ERRNO, PF_DEC},
                                    {"data", PT_BYTEBUF, PF_NA},
                                    {"fd", PT_FD, PF_DEC},
                                    {"size", PT_UINT32, PF_DEC},
                                    {"tuple", PT_SOCKTUPLE, PF_NA}}},
        [PPME_SOCKET_SENDMMSG_E] = {"sendmmsg", EC_IO_WRITE | EC_SYSCALL, EF_OLD_VERSION, 0},
        [PPME_SOCKET_SENDMMSG_X] = {"sendmmsg",
                                    EC_IO_WRITE | EC_SYSCALL,
                                    EF_USES_FD | EF_WRITES_TO_FD | EF_MODIFIES_STATE |
                                            EF_CONVERTER_MANAGED,
                                    5,
                                    {{"res", PT_ERRNO, PF_DEC},
                                     {"fd", PT_FD, PF_DEC},
                                     {"size", PT_UINT32, PF_DEC},
                                     {"data", PT_BYTEBUF, PF_NA},
                                     {"tuple", PT_SOCKTUPLE, PF_NA}}},
        [PPME_SOCKET_RECVMSG_E] = {"recvmsg",
                                   EC_IO_READ | EC_SYSCALL,
                                   EF_OLD_VERSION | EF_USES_FD | EF_READS_FROM_FD |
                                           EF_MODIFIES_STATE | EF_CONVERTER_MANAGED,
                                   1,
                                   {{"fd", PT_FD, PF_DEC}}},
        [PPME_SOCKET_RECVMSG_X] = {"recvmsg",
                                   EC_IO_READ | EC_SYSCALL,
                                   EF_USES_FD | EF_READS_FROM_FD | EF_MODIFIES_STATE |
                                           EF_CONVERTER_MANAGED,
                                   6,
                                   {{"res", PT_ERRNO, PF_DEC},
                                    {"size", PT_UINT32, PF_DEC},
                                    {"data", PT_BYTEBUF, PF_NA},
                                    {"tuple", PT_SOCKTUPLE, PF_NA},
                                    {"msgcontrol", PT_BYTEBUF, PF_NA},
                                    {"fd", PT_FD, PF_DEC}}},
        [PPME_SOCKET_RECVMMSG_E] = {"recvmmsg", EC_IO_READ | EC_SYSCALL, EF_OLD_VERSION, 0},
        [PPME_SOCKET_RECVMMSG_X] = {"recvmmsg",
                                    EC_IO_READ | EC_SYSCALL,
                                    EF_USES_FD | EF_READS_FROM_FD | EF_MODIFIES_STATE |
                                            EF_CONVERTER_MANAGED,
                                    6,
                                    {{"res", PT_ERRNO, PF_DEC},
                                     {"fd", PT_FD, PF_DEC},
                                     {"size", PT_UINT32, PF_DEC},
                                     {"data", PT_BYTEBUF, PF_NA},
                                     {"tuple", PT_SOCKTUPLE, PF_NA},
                                     {"msgcontrol", PT_BYTEBUF, PF_NA}}},
        [PPME_SOCKET_ACCEPT4_E] = {"accept",
                                   EC_NET | EC_SYSCALL,
                                   EF_OLD_VERSION | EF_CREATES_FD | EF_MODIFIES_STATE |
                                           EF_CONVERTER_MANAGED,
                                   1,
                                   {{"flags", PT_INT32, PF_HEX}}},
        [PPME_SOCKET_ACCEPT4_X] = {"accept",
                                   EC_NET | EC_SYSCALL,
                                   EF_OLD_VERSION | EF_CREATES_FD | EF_MODIFIES_STATE |
                                           EF_CONVERTER_MANAGED,
                                   3,
                                   {{"fd", PT_FD, PF_DEC},
                                    {"tuple", PT_SOCKTUPLE, PF_NA},
                                    {"queuepct", PT_UINT8, PF_DEC}}},
        [PPME_SYSCALL_CREAT_E] = {"creat",
                                  EC_FILE | EC_SYSCALL,
                                  EF_CREATES_FD | EF_MODIFIES_STATE | EF_CONVERTER_MANAGED,
                                  2,
                                  {{"name", PT_FSPATH, PF_NA}, {"mode", PT_UINT32, PF_OCT}}},
        [PPME_SYSCALL_CREAT_X] = {"creat",
                                  EC_FILE | EC_SYSCALL,
                                  EF_CREATES_FD | EF_MODIFIES_STATE | EF_CONVERTER_MANAGED,
                                  6,
                                  {{"fd", PT_FD, PF_DEC},
                                   {"name", PT_FSPATH, PF_NA},
                                   {"mode", PT_UINT32, PF_OCT},
                                   {"dev", PT_UINT32, PF_HEX},
                                   {"ino", PT_UINT64, PF_DEC},
                                   {"creat_flags", PT_FLAGS16, PF_HEX, creat_flags}}},
        [PPME_SYSCALL_PIPE_E] = {"pipe",
                                 EC_IPC | EC_SYSCALL,
                                 EF_OLD_VERSION | EF_CREATES_FD | EF_MODIFIES_STATE,
                                 0},
        [PPME_SYSCALL_PIPE_X] = {"pipe",
                                 EC_IPC | EC_SYSCALL,
                                 EF_CREATES_FD | EF_MODIFIES_STATE,
                                 4,
                                 {{"res", PT_ERRNO, PF_DEC},
                                  {"fd1", PT_FD, PF_DEC},
                                  {"fd2", PT_FD, PF_DEC},
                                  {"ino", PT_UINT64, PF_DEC}}},
        [PPME_SYSCALL_EVENTFD_E] = {"eventfd",
                                    EC_IPC | EC_SYSCALL,
                                    EF_OLD_VERSION | EF_CREATES_FD | EF_MODIFIES_STATE |
                                            EF_CONVERTER_MANAGED,
                                    2,
                                    {{"initval", PT_UINT64, PF_DEC}, {"flags", PT_UINT32, PF_HEX}}},
        [PPME_SYSCALL_EVENTFD_X] = {"eventfd",
                                    EC_IPC | EC_SYSCALL,
                                    EF_CREATES_FD | EF_MODIFIES_STATE | EF_CONVERTER_MANAGED,
                                    3,
                                    {{"res", PT_FD, PF_DEC},
                                     {"initval", PT_UINT64, PF_DEC},
                                     {"flags", PT_UINT32, PF_HEX}}},
        [PPME_SYSCALL_FUTEX_E] = {"futex",
                                  EC_IPC | EC_SYSCALL,
                                  EF_OLD_VERSION | EF_CONVERTER_MANAGED,
                                  3,
                                  {{"addr", PT_UINT64, PF_HEX},
                                   {"op", PT_FLAGS16, PF_HEX, futex_operations},
                                   {"val", PT_UINT64, PF_DEC}}},
        [PPME_SYSCALL_FUTEX_X] = {"futex",
                                  EC_IPC | EC_SYSCALL,
                                  EF_CONVERTER_MANAGED,
                                  4,
                                  {{"res", PT_ERRNO, PF_DEC},
                                   {"addr", PT_UINT64, PF_HEX},
                                   {"op", PT_FLAGS16, PF_HEX, futex_operations},
                                   {"val", PT_UINT64, PF_DEC}}},
        [PPME_SYSCALL_STAT_E] = {"stat", EC_FILE | EC_SYSCALL, EF_OLD_VERSION, 0},
        [PPME_SYSCALL_STAT_X] = {"stat",
                                 EC_FILE | EC_SYSCALL,
                                 EF_NONE,
                                 2,
                                 {{"res", PT_ERRNO, PF_DEC}, {"path", PT_FSPATH, PF_NA}}},
        [PPME_SYSCALL_LSTAT_E] = {"lstat", EC_FILE | EC_SYSCALL, EF_OLD_VERSION, 0},
        [PPME_SYSCALL_LSTAT_X] = {"lstat",
                                  EC_FILE | EC_SYSCALL,
                                  EF_NONE,
                                  2,
                                  {{"res", PT_ERRNO, PF_DEC}, {"path", PT_FSPATH, PF_NA}}},
        [PPME_SYSCALL_FSTAT_E] = {"fstat",
                                  EC_FILE | EC_SYSCALL,
                                  EF_OLD_VERSION | EF_USES_FD | EF_CONVERTER_MANAGED,
                                  1,
                                  {{"fd", PT_FD, PF_NA}}},
        [PPME_SYSCALL_FSTAT_X] = {"fstat",
                                  EC_FILE | EC_SYSCALL,
                                  EF_USES_FD | EF_CONVERTER_MANAGED,
                                  2,
                                  {{"res", PT_ERRNO, PF_DEC}, {"fd", PT_FD, PF_NA}}},
        [PPME_SYSCALL_STAT64_E] = {"stat64", EC_FILE | EC_SYSCALL, EF_OLD_VERSION, 0},
        [PPME_SYSCALL_STAT64_X] = {"stat64",
                                   EC_FILE | EC_SYSCALL,
                                   EF_NONE,
                                   2,
                                   {{"res", PT_ERRNO, PF_DEC}, {"path", PT_FSPATH, PF_NA}}},
        [PPME_SYSCALL_LSTAT64_E] = {"lstat64", EC_FILE | EC_SYSCALL, EF_OLD_VERSION, 0},
        [PPME_SYSCALL_LSTAT64_X] = {"lstat64",
                                    EC_FILE | EC_SYSCALL,
                                    EF_NONE,
                                    2,
                                    {{"res", PT_ERRNO, PF_DEC}, {"path", PT_FSPATH, PF_NA}}},
        [PPME_SYSCALL_FSTAT64_E] = {"fstat64",
                                    EC_FILE | EC_SYSCALL,
                                    EF_OLD_VERSION | EF_USES_FD,
                                    1,
                                    {{"fd", PT_FD, PF_NA}}},
        [PPME_SYSCALL_FSTAT64_X] =
                {"fstat64", EC_FILE | EC_SYSCALL, EF_NONE, 1, {{"res", PT_ERRNO, PF_DEC}}},
        [PPME_SYSCALL_EPOLLWAIT_E] = {"epoll_wait",
                                      EC_WAIT | EC_SYSCALL,
                                      EF_OLD_VERSION | EF_WAITS | EF_CONVERTER_MANAGED,
                                      1,
                                      {{"maxevents", PT_ERRNO, PF_DEC}}},
        [PPME_SYSCALL_EPOLLWAIT_X] = {"epoll_wait",
                                      EC_WAIT | EC_SYSCALL,
                                      EF_WAITS | EF_CONVERTER_MANAGED,
                                      2,
                                      {{"res", PT_ERRNO, PF_DEC}, {"maxevents", PT_ERRNO, PF_DEC}}},
        [PPME_SYSCALL_POLL_E] = {"poll",
                                 EC_WAIT | EC_SYSCALL,
                                 EF_OLD_VERSION | EF_WAITS | EF_CONVERTER_MANAGED,
                                 2,
                                 {{"fds", PT_FDLIST, PF_DEC}, {"timeout", PT_INT64, PF_DEC}}},
        [PPME_SYSCALL_POLL_X] = {"poll",
                                 EC_WAIT | EC_SYSCALL,
                                 EF_WAITS | EF_CONVERTER_MANAGED,
                                 3,
                                 {{"res", PT_ERRNO, PF_DEC},
                                  {"fds", PT_FDLIST, PF_DEC},
                                  {"timeout", PT_INT64, PF_DEC}}},
        [PPME_SYSCALL_SELECT_E] = {"select", EC_WAIT | EC_SYSCALL, EF_OLD_VERSION | EF_WAITS, 0},
        [PPME_SYSCALL_SELECT_X] =
                {"select", EC_WAIT | EC_SYSCALL, EF_WAITS, 1, {{"res", PT_ERRNO, PF_DEC}}},
        [PPME_SYSCALL_NEWSELECT_E] = {"select", EC_WAIT | EC_SYSCALL, EF_OLD_VERSION | EF_WAITS, 0},
        [PPME_SYSCALL_NEWSELECT_X] = {"select",
                                      EC_WAIT | EC_SYSCALL,
                                      EF_OLD_VERSION | EF_WAITS | EF_CONVERTER_MANAGED,
                                      1,
                                      {{"res", PT_ERRNO, PF_DEC}}},
        [PPME_SYSCALL_LSEEK_E] = {"lseek",
                                  EC_FILE | EC_SYSCALL,
                                  EF_OLD_VERSION | EF_USES_FD | EF_CONVERTER_MANAGED,
                                  3,
                                  {{"fd", PT_FD, PF_DEC},
                                   {"offset", PT_UINT64, PF_DEC},
                                   {"whence", PT_ENUMFLAGS8, PF_DEC, lseek_whence}}},
        [PPME_SYSCALL_LSEEK_X] = {"lseek",
                                  EC_FILE | EC_SYSCALL,
                                  EF_USES_FD | EF_CONVERTER_MANAGED,
                                  4,
                                  {{"res", PT_ERRNO, PF_DEC},
                                   {"fd", PT_FD, PF_DEC},
                                   {"offset", PT_UINT64, PF_DEC},
                                   {"whence", PT_ENUMFLAGS8, PF_DEC, lseek_whence}}},
        [PPME_SYSCALL_LLSEEK_E] = {"llseek",
                                   EC_FILE | EC_SYSCALL,
                                   EF_OLD_VERSION | EF_USES_FD | EF_CONVERTER_MANAGED,
                                   3,
                                   {{"fd", PT_FD, PF_DEC},
                                    {"offset", PT_UINT64, PF_DEC},
                                    {"whence", PT_ENUMFLAGS8, PF_DEC, lseek_whence}}},
        [PPME_SYSCALL_LLSEEK_X] = {"llseek",
                                   EC_FILE | EC_SYSCALL,
                                   EF_USES_FD | EF_CONVERTER_MANAGED,
                                   4,
                                   {{"res", PT_ERRNO, PF_DEC},
                                    {"fd", PT_FD, PF_DEC},
                                    {"offset", PT_UINT64, PF_DEC},
                                    {"whence", PT_ENUMFLAGS8, PF_DEC, lseek_whence}}},
        [PPME_SYSCALL_IOCTL_2_E] = {"ioctl",
                                    EC_IO_OTHER | EC_SYSCALL,
                                    EF_OLD_VERSION | EF_USES_FD | EF_CONVERTER_MANAGED,
                                    2,
                                    {{"fd", PT_FD, PF_DEC}, {"request", PT_UINT64, PF_HEX}}},
        [PPME_SYSCALL_IOCTL_2_X] = {"ioctl",
                                    EC_IO_OTHER | EC_SYSCALL,
                                    EF_OLD_VERSION | EF_USES_FD | EF_CONVERTER_MANAGED,
                                    1,
                                    {{"res", PT_ERRNO, PF_DEC}}},
        [PPME_SYSCALL_GETCWD_E] = {"getcwd", EC_FILE | EC_SYSCALL, EF_OLD_VERSION, 0},
        /* Note: path is PT_CHARBUF and not PT_FSPATH because we assume it's absolute and will never
           need resolution */
        [PPME_SYSCALL_GETCWD_X] = {"getcwd",
                                   EC_FILE | EC_SYSCALL,
                                   EF_NONE,
                                   2,
                                   {{"res", PT_ERRNO, PF_DEC}, {"path", PT_CHARBUF, PF_NA}}},
        /* Note: path is PT_CHARBUF and not PT_FSPATH because we don't want it to be resolved, since
           the event handler already changes it */
        [PPME_SYSCALL_CHDIR_E] = {"chdir",
                                  EC_FILE | EC_SYSCALL,
                                  EF_OLD_VERSION | EF_MODIFIES_STATE,
                                  0},
        [PPME_SYSCALL_CHDIR_X] = {"chdir",
                                  EC_FILE | EC_SYSCALL,
                                  EF_MODIFIES_STATE,
                                  2,
                                  {{"res", PT_ERRNO, PF_DEC}, {"path", PT_CHARBUF, PF_NA}}},
        [PPME_SYSCALL_FCHDIR_E] = {"fchdir",
                                   EC_FILE | EC_SYSCALL,
                                   EF_OLD_VERSION | EF_USES_FD | EF_MODIFIES_STATE |
                                           EF_CONVERTER_MANAGED,
                                   1,
                                   {{"fd", PT_FD, PF_NA}}},
        [PPME_SYSCALL_FCHDIR_X] = {"fchdir",
                                   EC_FILE | EC_SYSCALL,
                                   EF_USES_FD | EF_MODIFIES_STATE | EF_CONVERTER_MANAGED,
                                   2,
                                   {{"res", PT_ERRNO, PF_DEC}, {"fd", PT_FD, PF_NA}}},
        [PPME_SYSCALL_MKDIR_E] = {"mkdir",
                                  EC_FILE | EC_SYSCALL,
                                  EF_OLD_VERSION | EF_CONVERTER_MANAGED,
                                  2,
                                  {{"path", PT_FSPATH, PF_NA}, {"mode", PT_UINT32, PF_HEX}}},
        [PPME_SYSCALL_MKDIR_X] = {"mkdir",
                                  EC_FILE | EC_SYSCALL,
                                  EF_OLD_VERSION | EF_CONVERTER_MANAGED,
                                  1,
                                  {{"res", PT_ERRNO, PF_DEC}}},
        [PPME_SYSCALL_RMDIR_E] = {"rmdir",
                                  EC_FILE | EC_SYSCALL,
                                  EF_OLD_VERSION | EF_CONVERTER_MANAGED,
                                  1,
                                  {{"path", PT_FSPATH, PF_NA}}},
        [PPME_SYSCALL_RMDIR_X] = {"rmdir",
                                  EC_FILE | EC_SYSCALL,
                                  EF_OLD_VERSION | EF_CONVERTER_MANAGED,
                                  1,
                                  {{"res", PT_ERRNO, PF_DEC}}},
        [PPME_SYSCALL_OPENAT_E] = {"openat",
                                   EC_FILE | EC_SYSCALL,
                                   EF_OLD_VERSION | EF_CREATES_FD | EF_MODIFIES_STATE |
                                           EF_CONVERTER_MANAGED,
                                   4,
                                   {{"dirfd", PT_FD, PF_DEC},
                                    {"name", PT_CHARBUF, PF_NA},
                                    {"flags", PT_FLAGS32, PF_HEX, file_flags},
                                    {"mode", PT_UINT32, PF_OCT}}},
        [PPME_SYSCALL_OPENAT_X] = {"openat",
                                   EC_FILE | EC_SYSCALL,
                                   EF_OLD_VERSION | EF_CREATES_FD | EF_MODIFIES_STATE |
                                           EF_CONVERTER_MANAGED,
                                   1,
                                   {{"fd", PT_FD, PF_DEC}}},
        [PPME_SYSCALL_LINK_E] = {"link",
                                 EC_FILE | EC_SYSCALL,
                                 EF_OLD_VERSION | EF_CONVERTER_MANAGED,
                                 2,
                                 {{"oldpath", PT_FSPATH, PF_NA}, {"newpath", PT_FSPATH, PF_NA}}},
        [PPME_SYSCALL_LINK_X] = {"link",
                                 EC_FILE | EC_SYSCALL,
                                 EF_OLD_VERSION | EF_CONVERTER_MANAGED,
                                 1,
                                 {{"res", PT_ERRNO, PF_DEC}}},
        [PPME_SYSCALL_LINKAT_E] = {"linkat",
                                   EC_FILE | EC_SYSCALL,
                                   EF_OLD_VERSION | EF_CONVERTER_MANAGED,
                                   4,
                                   {{"olddir", PT_FD, PF_DEC},
                                    {"oldpath", PT_CHARBUF, PF_NA},
                                    {"newdir", PT_FD, PF_DEC},
                                    {"newpath", PT_CHARBUF, PF_NA}}},
        [PPME_SYSCALL_LINKAT_X] = {"linkat",
                                   EC_FILE | EC_SYSCALL,
                                   EF_OLD_VERSION | EF_CONVERTER_MANAGED,
                                   1,
                                   {{"res", PT_ERRNO, PF_DEC}}},
        [PPME_SYSCALL_UNLINK_E] = {"unlink",
                                   EC_FILE | EC_SYSCALL,
                                   EF_OLD_VERSION | EF_CONVERTER_MANAGED,
                                   1,
                                   {{"path", PT_FSPATH, PF_NA}}},
        [PPME_SYSCALL_UNLINK_X] = {"unlink",
                                   EC_FILE | EC_SYSCALL,
                                   EF_OLD_VERSION | EF_CONVERTER_MANAGED,
                                   1,
                                   {{"res", PT_ERRNO, PF_DEC}}},
        [PPME_SYSCALL_UNLINKAT_E] = {"unlinkat",
                                     EC_FILE | EC_SYSCALL,
                                     EF_OLD_VERSION | EF_CONVERTER_MANAGED,
                                     2,
                                     {{"dirfd", PT_FD, PF_DEC}, {"name", PT_CHARBUF, PF_NA}}},
        [PPME_SYSCALL_UNLINKAT_X] = {"unlinkat",
                                     EC_FILE | EC_SYSCALL,
                                     EF_OLD_VERSION | EF_CONVERTER_MANAGED,
                                     1,
                                     {{"res", PT_ERRNO, PF_DEC}}},
        [PPME_SYSCALL_PREAD_E] =
                {"pread",
                 EC_IO_READ | EC_SYSCALL,
                 EF_OLD_VERSION | EF_USES_FD | EF_READS_FROM_FD | EF_CONVERTER_MANAGED,
                 3,
                 {{"fd", PT_FD, PF_DEC}, {"size", PT_UINT32, PF_DEC}, {"pos", PT_UINT64, PF_DEC}}},
        [PPME_SYSCALL_PREAD_X] = {"pread",
                                  EC_IO_READ | EC_SYSCALL,
                                  EF_USES_FD | EF_READS_FROM_FD | EF_CONVERTER_MANAGED,
                                  5,
                                  {{"res", PT_ERRNO, PF_DEC},
                                   {"data", PT_BYTEBUF, PF_NA},
                                   {"fd", PT_FD, PF_DEC},
                                   {"size", PT_UINT32, PF_DEC},
                                   {"pos", PT_UINT64, PF_DEC}}},
        [PPME_SYSCALL_PWRITE_E] =
                {"pwrite",
                 EC_IO_WRITE | EC_SYSCALL,
                 EF_OLD_VERSION | EF_USES_FD | EF_WRITES_TO_FD | EF_CONVERTER_MANAGED,
                 3,
                 {{"fd", PT_FD, PF_DEC}, {"size", PT_UINT32, PF_DEC}, {"pos", PT_UINT64, PF_DEC}}},
        [PPME_SYSCALL_PWRITE_X] = {"pwrite",
                                   EC_IO_WRITE | EC_SYSCALL,
                                   EF_USES_FD | EF_WRITES_TO_FD | EF_CONVERTER_MANAGED,
                                   5,
                                   {{"res", PT_ERRNO, PF_DEC},
                                    {"data", PT_BYTEBUF, PF_NA},
                                    {"fd", PT_FD, PF_DEC},
                                    {"size", PT_UINT32, PF_DEC},
                                    {"pos", PT_UINT64, PF_DEC}}},
        [PPME_SYSCALL_READV_E] = {"readv",
                                  EC_IO_READ | EC_SYSCALL,
                                  EF_OLD_VERSION | EF_USES_FD | EF_READS_FROM_FD |
                                          EF_CONVERTER_MANAGED,
                                  1,
                                  {{"fd", PT_FD, PF_DEC}}},
        [PPME_SYSCALL_READV_X] = {"readv",
                                  EC_IO_READ | EC_SYSCALL,
                                  EF_USES_FD | EF_READS_FROM_FD | EF_CONVERTER_MANAGED,
                                  4,
                                  {{"res", PT_ERRNO, PF_DEC},
                                   {"size", PT_UINT32, PF_DEC},
                                   {"data", PT_BYTEBUF, PF_NA},
                                   {"fd", PT_FD, PF_DEC}}},
        [PPME_SYSCALL_WRITEV_E] = {"writev",
                                   EC_IO_WRITE | EC_SYSCALL,
                                   EF_OLD_VERSION | EF_USES_FD | EF_WRITES_TO_FD |
                                           EF_CONVERTER_MANAGED,
                                   2,
                                   {{"fd", PT_FD, PF_DEC}, {"size", PT_UINT32, PF_DEC}}},
        [PPME_SYSCALL_WRITEV_X] = {"writev",
                                   EC_IO_WRITE | EC_SYSCALL,
                                   EF_USES_FD | EF_WRITES_TO_FD | EF_CONVERTER_MANAGED,
                                   4,
                                   {{"res", PT_ERRNO, PF_DEC},
                                    {"data", PT_BYTEBUF, PF_NA},
                                    {"fd", PT_FD, PF_DEC},
                                    {"size", PT_UINT32, PF_DEC}}},
        [PPME_SYSCALL_PREADV_E] = {"preadv",
                                   EC_IO_READ | EC_SYSCALL,
                                   EF_OLD_VERSION | EF_USES_FD | EF_READS_FROM_FD |
                                           EF_CONVERTER_MANAGED,
                                   2,
                                   {{"fd", PT_FD, PF_DEC}, {"pos", PT_UINT64, PF_DEC}}},
        [PPME_SYSCALL_PREADV_X] = {"preadv",
                                   EC_IO_READ | EC_SYSCALL,
                                   EF_USES_FD | EF_READS_FROM_FD | EF_CONVERTER_MANAGED,
                                   5,
                                   {{"res", PT_ERRNO, PF_DEC},
                                    {"size", PT_UINT32, PF_DEC},
                                    {"data", PT_BYTEBUF, PF_NA},
                                    {"fd", PT_FD, PF_DEC},
                                    {"pos", PT_UINT64, PF_DEC}}},
        [PPME_SYSCALL_PWRITEV_E] =
                {"pwritev",
                 EC_IO_WRITE | EC_SYSCALL,
                 EF_OLD_VERSION | EF_USES_FD | EF_WRITES_TO_FD | EF_CONVERTER_MANAGED,
                 3,
                 {{"fd", PT_FD, PF_DEC}, {"size", PT_UINT32, PF_DEC}, {"pos", PT_UINT64, PF_DEC}}},
        [PPME_SYSCALL_PWRITEV_X] = {"pwritev",
                                    EC_IO_WRITE | EC_SYSCALL,
                                    EF_USES_FD | EF_WRITES_TO_FD | EF_CONVERTER_MANAGED,
                                    5,
                                    {{"res", PT_ERRNO, PF_DEC},
                                     {"data", PT_BYTEBUF, PF_NA},
                                     {"fd", PT_FD, PF_DEC},
                                     {"size", PT_UINT32, PF_DEC},
                                     {"pos", PT_UINT64, PF_DEC}}},
        [PPME_SYSCALL_DUP_E] = {"dup",
                                EC_IO_OTHER | EC_SYSCALL,
                                EF_OLD_VERSION | EF_CREATES_FD | EF_USES_FD | EF_MODIFIES_STATE |
                                        EF_CONVERTER_MANAGED,
                                1,
                                {{"fd", PT_FD, PF_DEC}}},
        [PPME_SYSCALL_DUP_X] = {"dup",
                                EC_IO_OTHER | EC_SYSCALL,
                                EF_OLD_VERSION | EF_CREATES_FD | EF_USES_FD | EF_MODIFIES_STATE |
                                        EF_CONVERTER_MANAGED,
                                1,
                                {{"res", PT_FD, PF_DEC}}},
        [PPME_SYSCALL_SIGNALFD_E] =
                {"signalfd",
                 EC_SIGNAL | EC_SYSCALL,
                 EF_OLD_VERSION | EF_CREATES_FD | EF_MODIFIES_STATE | EF_CONVERTER_MANAGED,
                 3,
                 {{"fd", PT_FD, PF_DEC}, {"mask", PT_UINT32, PF_HEX}, {"flags", PT_UINT8, PF_HEX}}},
        [PPME_SYSCALL_SIGNALFD_X] = {"signalfd",
                                     EC_SIGNAL | EC_SYSCALL,
                                     EF_CREATES_FD | EF_MODIFIES_STATE | EF_CONVERTER_MANAGED,
                                     4,
                                     {{"res", PT_FD, PF_DEC},
                                      {"fd", PT_FD, PF_DEC},
                                      {"mask", PT_UINT32, PF_HEX},
                                      {"flags", PT_UINT8, PF_HEX}}},
        [PPME_SYSCALL_KILL_E] = {"kill",
                                 EC_SIGNAL | EC_SYSCALL,
                                 EF_OLD_VERSION | EF_CONVERTER_MANAGED,
                                 2,
                                 {{"pid", PT_PID, PF_DEC}, {"sig", PT_SIGTYPE, PF_DEC}}},
        [PPME_SYSCALL_KILL_X] = {"kill",
                                 EC_SIGNAL | EC_SYSCALL,
                                 EF_CONVERTER_MANAGED,
                                 3,
                                 {{"res", PT_ERRNO, PF_DEC},
                                  {"pid", PT_PID, PF_DEC},
                                  {"sig", PT_SIGTYPE, PF_DEC}}},
        [PPME_SYSCALL_TKILL_E] = {"tkill",
                                  EC_SIGNAL | EC_SYSCALL,
                                  EF_OLD_VERSION | EF_CONVERTER_MANAGED,
                                  2,
                                  {{"tid", PT_PID, PF_DEC}, {"sig", PT_SIGTYPE, PF_DEC}}},
        [PPME_SYSCALL_TKILL_X] = {"tkill",
                                  EC_SIGNAL | EC_SYSCALL,
                                  EF_CONVERTER_MANAGED,
                                  3,
                                  {{"res", PT_ERRNO, PF_DEC},
                                   {"tid", PT_PID, PF_DEC},
                                   {"sig", PT_SIGTYPE, PF_DEC}}},
        [PPME_SYSCALL_TGKILL_E] = {"tgkill",
                                   EC_SIGNAL | EC_SYSCALL,
                                   EF_OLD_VERSION | EF_CONVERTER_MANAGED,
                                   3,
                                   {{"pid", PT_PID, PF_DEC},
                                    {"tid", PT_PID, PF_DEC},
                                    {"sig", PT_SIGTYPE, PF_DEC}}},
        [PPME_SYSCALL_TGKILL_X] = {"tgkill",
                                   EC_SIGNAL | EC_SYSCALL,
                                   EF_CONVERTER_MANAGED,
                                   4,
                                   {{"res", PT_ERRNO, PF_DEC},
                                    {"pid", PT_PID, PF_DEC},
                                    {"tid", PT_PID, PF_DEC},
                                    {"sig", PT_SIGTYPE, PF_DEC}}},
        [PPME_SYSCALL_NANOSLEEP_E] = {"nanosleep",
                                      EC_SLEEP | EC_SYSCALL,
                                      EF_OLD_VERSION | EF_WAITS | EF_CONVERTER_MANAGED,
                                      1,
                                      {{"interval", PT_RELTIME, PF_DEC}}},
        [PPME_SYSCALL_NANOSLEEP_X] = {"nanosleep",
                                      EC_SLEEP | EC_SYSCALL,
                                      EF_WAITS | EF_CONVERTER_MANAGED,
                                      2,
                                      {{"res", PT_ERRNO, PF_DEC},
                                       {"interval", PT_RELTIME, PF_DEC}}},
        [PPME_SYSCALL_TIMERFD_CREATE_E] = {"timerfd_create",
                                           EC_TIME | EC_SYSCALL,
                                           EF_OLD_VERSION | EF_CREATES_FD | EF_MODIFIES_STATE |
                                                   EF_CONVERTER_MANAGED,
                                           2,
                                           {{"clockid", PT_UINT8, PF_DEC},
                                            {"flags", PT_UINT8, PF_HEX}}},
        [PPME_SYSCALL_TIMERFD_CREATE_X] = {"timerfd_create",
                                           EC_TIME | EC_SYSCALL,
                                           EF_CREATES_FD | EF_MODIFIES_STATE | EF_CONVERTER_MANAGED,
                                           3,
                                           {{"res", PT_FD, PF_DEC},
                                            {"clockid", PT_UINT8, PF_DEC},
                                            {"flags", PT_UINT8, PF_HEX}}},
        [PPME_SYSCALL_INOTIFY_INIT_E] = {"inotify_init",
                                         EC_IPC | EC_SYSCALL,
                                         EF_OLD_VERSION | EF_CREATES_FD | EF_MODIFIES_STATE |
                                                 EF_CONVERTER_MANAGED,
                                         1,
                                         {{"flags", PT_UINT8, PF_HEX}}},
        [PPME_SYSCALL_INOTIFY_INIT_X] = {"inotify_init",
                                         EC_IPC | EC_SYSCALL,
                                         EF_CREATES_FD | EF_MODIFIES_STATE | EF_CONVERTER_MANAGED,
                                         2,
                                         {{"res", PT_FD, PF_DEC}, {"flags", PT_UINT8, PF_HEX}}},
        [PPME_SYSCALL_GETRLIMIT_E] = {"getrlimit",
                                      EC_PROCESS | EC_SYSCALL,
                                      EF_OLD_VERSION | EF_CONVERTER_MANAGED,
                                      1,
                                      {{"resource", PT_ENUMFLAGS8, PF_DEC, rlimit_resources}}},
        [PPME_SYSCALL_GETRLIMIT_X] = {"getrlimit",
                                      EC_PROCESS | EC_SYSCALL,
                                      EF_CONVERTER_MANAGED,
                                      4,
                                      {{"res", PT_ERRNO, PF_DEC},
                                       {"cur", PT_INT64, PF_DEC},
                                       {"max", PT_INT64, PF_DEC},
                                       {"resource", PT_ENUMFLAGS8, PF_DEC, rlimit_resources}}},
        [PPME_SYSCALL_SETRLIMIT_E] = {"setrlimit",
                                      EC_PROCESS | EC_SYSCALL,
                                      EF_OLD_VERSION | EF_CONVERTER_MANAGED,
                                      1,
                                      {{"resource", PT_ENUMFLAGS8, PF_DEC, rlimit_resources}}},
        [PPME_SYSCALL_SETRLIMIT_X] = {"setrlimit",
                                      EC_PROCESS | EC_SYSCALL,
                                      EF_CONVERTER_MANAGED,
                                      4,
                                      {{"res", PT_ERRNO, PF_DEC},
                                       {"cur", PT_INT64, PF_DEC},
                                       {"max", PT_INT64, PF_DEC},
                                       {"resource", PT_ENUMFLAGS8, PF_DEC, rlimit_resources}}},
        [PPME_SYSCALL_PRLIMIT_E] = {"prlimit",
                                    EC_PROCESS | EC_SYSCALL,
                                    EF_OLD_VERSION | EF_CONVERTER_MANAGED,
                                    2,
                                    {{"pid", PT_PID, PF_DEC},
                                     {"resource", PT_ENUMFLAGS8, PF_DEC, rlimit_resources}}},
        [PPME_SYSCALL_PRLIMIT_X] = {"prlimit",
                                    EC_PROCESS | EC_SYSCALL,
                                    EF_CONVERTER_MANAGED,
                                    7,
                                    {{"res", PT_ERRNO, PF_DEC},
                                     {"newcur", PT_INT64, PF_DEC},
                                     {"newmax", PT_INT64, PF_DEC},
                                     {"oldcur", PT_INT64, PF_DEC},
                                     {"oldmax", PT_INT64, PF_DEC},
                                     {"pid", PT_INT64, PF_DEC},
                                     {"resource", PT_ENUMFLAGS8, PF_DEC, rlimit_resources}}},
        [PPME_SCHEDSWITCH_1_E] = {"switch",
                                  EC_SCHEDULER | EC_TRACEPOINT,
                                  EF_OLD_VERSION | EF_SKIPPARSERESET | EF_CONVERTER_MANAGED,
                                  1,
                                  {{"next", PT_PID, PF_DEC}}},
        [PPME_SCHEDSWITCH_1_X] = {"NA", EC_UNKNOWN, EF_UNUSED, 0},
        [PPME_DROP_E] = {"drop",
                         EC_INTERNAL | EC_METAEVENT,
                         EF_SKIPPARSERESET,
                         1,
                         {{"ratio", PT_UINT32, PF_DEC}}},
        [PPME_DROP_X] = {"drop",
                         EC_INTERNAL | EC_METAEVENT,
                         EF_SKIPPARSERESET,
                         1,
                         {{"ratio", PT_UINT32, PF_DEC}}},
        [PPME_SYSCALL_FCNTL_E] = {"fcntl",
                                  EC_IO_OTHER | EC_SYSCALL,
                                  EF_OLD_VERSION | EF_USES_FD | EF_MODIFIES_STATE |
                                          EF_CONVERTER_MANAGED,
                                  2,
                                  {{"fd", PT_FD, PF_DEC},
                                   {"cmd", PT_ENUMFLAGS8, PF_DEC, fcntl_commands}}},
        [PPME_SYSCALL_FCNTL_X] = {"fcntl",
                                  EC_IO_OTHER | EC_SYSCALL,
                                  EF_USES_FD | EF_MODIFIES_STATE | EF_CONVERTER_MANAGED,
                                  3,
                                  {{"res", PT_FD, PF_DEC},
                                   {"fd", PT_FD, PF_DEC},
                                   {"cmd", PT_ENUMFLAGS8, PF_DEC, fcntl_commands}}},
        [PPME_SCHEDSWITCH_6_E] =
                {"switch",
                 EC_SCHEDULER | EC_TRACEPOINT,
                 EF_NONE,
                 6,
                 {{"next", PT_PID, PF_DEC},
                  {"pgft_maj", PT_UINT64, PF_DEC},
                  {"pgft_min", PT_UINT64, PF_DEC},
                  {"vm_size", PT_UINT32, PF_DEC},
                  {"vm_rss", PT_UINT32, PF_DEC},
                  {"vm_swap", PT_UINT32, PF_DEC}}},  /// TODO: do we need SKIPPARSERESET flag?
        [PPME_SCHEDSWITCH_6_X] = {"NA", EC_UNKNOWN, EF_UNUSED, 0},
        [PPME_SYSCALL_EXECVE_13_E] = {"execve",
                                      EC_PROCESS | EC_SYSCALL,
                                      EF_OLD_VERSION | EF_MODIFIES_STATE | EF_CONVERTER_MANAGED,
                                      0},
        [PPME_SYSCALL_EXECVE_13_X] = {"execve",
                                      EC_PROCESS | EC_SYSCALL,
                                      EF_OLD_VERSION | EF_MODIFIES_STATE | EF_CONVERTER_MANAGED,
                                      13,
                                      {{"res", PT_ERRNO, PF_DEC},
                                       {"exe", PT_CHARBUF, PF_NA},
                                       {"args", PT_BYTEBUF, PF_NA},
                                       {"tid", PT_PID, PF_DEC},
                                       {"pid", PT_PID, PF_DEC},
                                       {"ptid", PT_PID, PF_DEC},
                                       {"cwd", PT_CHARBUF, PF_NA},
                                       {"fdlimit", PT_UINT64, PF_DEC},
                                       {"pgft_maj", PT_UINT64, PF_DEC},
                                       {"pgft_min", PT_UINT64, PF_DEC},
                                       {"vm_size", PT_UINT32, PF_DEC},
                                       {"vm_rss", PT_UINT32, PF_DEC},
                                       {"vm_swap", PT_UINT32, PF_DEC}}},
        [PPME_SYSCALL_CLONE_16_E] = {"clone",
                                     EC_PROCESS | EC_SYSCALL,
                                     EF_OLD_VERSION | EF_MODIFIES_STATE,
                                     0},
        [PPME_SYSCALL_CLONE_16_X] = {"clone",
                                     EC_PROCESS | EC_SYSCALL,
                                     EF_OLD_VERSION | EF_MODIFIES_STATE | EF_CONVERTER_MANAGED,
                                     16,
                                     {{"res", PT_PID, PF_DEC},
                                      {"exe", PT_CHARBUF, PF_NA},
                                      {"args", PT_BYTEBUF, PF_NA},
                                      {"tid", PT_PID, PF_DEC},
                                      {"pid", PT_PID, PF_DEC},
                                      {"ptid", PT_PID, PF_DEC},
                                      {"cwd", PT_CHARBUF, PF_NA},
                                      {"fdlimit", PT_INT64, PF_DEC},
                                      {"pgft_maj", PT_UINT64, PF_DEC},
                                      {"pgft_min", PT_UINT64, PF_DEC},
                                      {"vm_size", PT_UINT32, PF_DEC},
                                      {"vm_rss", PT_UINT32, PF_DEC},
                                      {"vm_swap", PT_UINT32, PF_DEC},
                                      {"flags", PT_FLAGS32, PF_HEX, clone_flags},
                                      {"uid", PT_UINT32, PF_DEC},
                                      {"gid", PT_UINT32, PF_DEC}}},
        [PPME_SYSCALL_BRK_4_E] = {"brk",
                                  EC_MEMORY | EC_SYSCALL,
                                  EF_OLD_VERSION | EF_CONVERTER_MANAGED,
                                  1,
                                  {{"addr", PT_UINT64, PF_HEX}}},
        [PPME_SYSCALL_BRK_4_X] = {"brk",
                                  EC_MEMORY | EC_SYSCALL,
                                  EF_CONVERTER_MANAGED,
                                  5,
                                  {{"res", PT_UINT64, PF_HEX},
                                   {"vm_size", PT_UINT32, PF_DEC},
                                   {"vm_rss", PT_UINT32, PF_DEC},
                                   {"vm_swap", PT_UINT32, PF_DEC},
                                   {"addr", PT_UINT64, PF_HEX}}},
        [PPME_SYSCALL_MMAP_E] = {"mmap",
                                 EC_MEMORY | EC_SYSCALL,
                                 EF_OLD_VERSION | EF_USES_FD | EF_CONVERTER_MANAGED,
                                 6,
                                 {{"addr", PT_UINT64, PF_HEX},
                                  {"length", PT_UINT64, PF_DEC},
                                  {"prot", PT_FLAGS32, PF_HEX, prot_flags},
                                  {"flags", PT_FLAGS32, PF_HEX, mmap_flags},
                                  {"fd", PT_FD, PF_DEC},
                                  {"offset", PT_UINT64, PF_DEC}}},
        [PPME_SYSCALL_MMAP_X] = {"mmap",
                                 EC_MEMORY | EC_SYSCALL,
                                 EF_USES_FD | EF_CONVERTER_MANAGED,
                                 10,
                                 {{"res", PT_ERRNO, PF_HEX},
                                  {"vm_size", PT_UINT32, PF_DEC},
                                  {"vm_rss", PT_UINT32, PF_DEC},
                                  {"vm_swap", PT_UINT32, PF_DEC},
                                  {"addr", PT_UINT64, PF_HEX},
                                  {"length", PT_UINT64, PF_DEC},
                                  {"prot", PT_FLAGS32, PF_HEX, prot_flags},
                                  {"flags", PT_FLAGS32, PF_HEX, mmap_flags},
                                  {"fd", PT_FD, PF_DEC},
                                  {"offset", PT_UINT64, PF_DEC}}},
        [PPME_SYSCALL_MMAP2_E] = {"mmap2",
                                  EC_MEMORY | EC_SYSCALL,
                                  EF_OLD_VERSION | EF_USES_FD | EF_CONVERTER_MANAGED,
                                  6,
                                  {{"addr", PT_UINT64, PF_HEX},
                                   {"length", PT_UINT64, PF_DEC},
                                   {"prot", PT_FLAGS32, PF_HEX, prot_flags},
                                   {"flags", PT_FLAGS32, PF_HEX, mmap_flags},
                                   {"fd", PT_FD, PF_DEC},
                                   {"pgoffset", PT_UINT64, PF_DEC}}},
        [PPME_SYSCALL_MMAP2_X] = {"mmap2",
                                  EC_MEMORY | EC_SYSCALL,
                                  EF_USES_FD | EF_CONVERTER_MANAGED,
                                  10,
                                  {{"res", PT_ERRNO, PF_HEX},
                                   {"vm_size", PT_UINT32, PF_DEC},
                                   {"vm_rss", PT_UINT32, PF_DEC},
                                   {"vm_swap", PT_UINT32, PF_DEC},
                                   {"addr", PT_UINT64, PF_HEX},
                                   {"length", PT_UINT64, PF_DEC},
                                   {"prot", PT_FLAGS32, PF_HEX, prot_flags},
                                   {"flags", PT_FLAGS32, PF_HEX, mmap_flags},
                                   {"fd", PT_FD, PF_DEC},
                                   {"pgoffset", PT_UINT64, PF_DEC}}},
        [PPME_SYSCALL_MUNMAP_E] = {"munmap",
                                   EC_MEMORY | EC_SYSCALL,
                                   EF_OLD_VERSION | EF_CONVERTER_MANAGED,
                                   2,
                                   {{"addr", PT_UINT64, PF_HEX}, {"length", PT_UINT64, PF_DEC}}},
        [PPME_SYSCALL_MUNMAP_X] = {"munmap",
                                   EC_MEMORY | EC_SYSCALL,
                                   EF_CONVERTER_MANAGED,
                                   6,
                                   {{"res", PT_ERRNO, PF_DEC},
                                    {"vm_size", PT_UINT32, PF_DEC},
                                    {"vm_rss", PT_UINT32, PF_DEC},
                                    {"vm_swap", PT_UINT32, PF_DEC},
                                    {"addr", PT_UINT64, PF_HEX},
                                    {"length", PT_UINT64, PF_DEC}}},
        [PPME_SYSCALL_SPLICE_E] = {"splice",
                                   EC_IO_OTHER | EC_SYSCALL,
                                   EF_OLD_VERSION | EF_USES_FD | EF_CONVERTER_MANAGED,
                                   4,
                                   {{"fd_in", PT_FD, PF_DEC},
                                    {"fd_out", PT_FD, PF_DEC},
                                    {"size", PT_UINT64, PF_DEC},
                                    {"flags", PT_FLAGS32, PF_HEX, splice_flags}}},
        [PPME_SYSCALL_SPLICE_X] = {"splice",
                                   EC_IO_OTHER | EC_SYSCALL,
                                   EF_USES_FD | EF_CONVERTER_MANAGED,
                                   5,
                                   {{"res", PT_ERRNO, PF_DEC},
                                    {"fd_in", PT_FD, PF_DEC},
                                    {"fd_out", PT_FD, PF_DEC},
                                    {"size", PT_UINT64, PF_DEC},
                                    {"flags", PT_FLAGS32, PF_HEX, splice_flags}}},
        [PPME_SYSCALL_PTRACE_E] = {"ptrace",
                                   EC_PROCESS | EC_SYSCALL,
                                   EF_OLD_VERSION | EF_CONVERTER_MANAGED,
                                   2,
                                   {{"request", PT_ENUMFLAGS16, PF_DEC, ptrace_requests},
                                    {"pid", PT_PID, PF_DEC}}},
        [PPME_SYSCALL_PTRACE_X] =
                {"ptrace",
                 EC_PROCESS | EC_SYSCALL,
                 EF_CONVERTER_MANAGED,
                 5,
                 {{"res", PT_ERRNO, PF_DEC},
                  {"addr", PT_DYN, PF_HEX, ptrace_dynamic_param, PPM_PTRACE_IDX_MAX},
                  {"data", PT_DYN, PF_HEX, ptrace_dynamic_param, PPM_PTRACE_IDX_MAX},
                  {"request", PT_ENUMFLAGS16, PF_DEC, ptrace_requests},
                  {"pid", PT_PID, PF_DEC}}},
        [PPME_SYSCALL_IOCTL_3_E] = {"ioctl",
                                    EC_IO_OTHER | EC_SYSCALL,
                                    EF_OLD_VERSION | EF_USES_FD | EF_CONVERTER_MANAGED,
                                    3,
                                    {{"fd", PT_FD, PF_DEC},
                                     {"request", PT_UINT64, PF_HEX},
                                     {"argument", PT_UINT64, PF_HEX}}},
        [PPME_SYSCALL_IOCTL_3_X] = {"ioctl",
                                    EC_IO_OTHER | EC_SYSCALL,
                                    EF_USES_FD | EF_CONVERTER_MANAGED,
                                    4,
                                    {{"res", PT_ERRNO, PF_DEC},
                                     {"fd", PT_FD, PF_DEC},
                                     {"request", PT_UINT64, PF_HEX},
                                     {"argument", PT_UINT64, PF_HEX}}},
        [PPME_SYSCALL_EXECVE_14_E] = {"execve",
                                      EC_PROCESS | EC_SYSCALL,
                                      EF_OLD_VERSION | EF_MODIFIES_STATE | EF_CONVERTER_MANAGED,
                                      0},
        [PPME_SYSCALL_EXECVE_14_X] = {"execve",
                                      EC_PROCESS | EC_SYSCALL,
                                      EF_OLD_VERSION | EF_MODIFIES_STATE | EF_CONVERTER_MANAGED,
                                      14,
                                      {{"res", PT_ERRNO, PF_DEC},
                                       {"exe", PT_CHARBUF, PF_NA},
                                       {"args", PT_BYTEBUF, PF_NA},
                                       {"tid", PT_PID, PF_DEC},
                                       {"pid", PT_PID, PF_DEC},
                                       {"ptid", PT_PID, PF_DEC},
                                       {"cwd", PT_CHARBUF, PF_NA},
                                       {"fdlimit", PT_UINT64, PF_DEC},
                                       {"pgft_maj", PT_UINT64, PF_DEC},
                                       {"pgft_min", PT_UINT64, PF_DEC},
                                       {"vm_size", PT_UINT32, PF_DEC},
                                       {"vm_rss", PT_UINT32, PF_DEC},
                                       {"vm_swap", PT_UINT32, PF_DEC},
                                       {"env", PT_BYTEBUF, PF_NA}}},
        [PPME_SYSCALL_RENAME_E] = {"rename", EC_FILE | EC_SYSCALL, EF_OLD_VERSION, 0},
        [PPME_SYSCALL_RENAME_X] = {"rename",
                                   EC_FILE | EC_SYSCALL,
                                   EF_NONE,
                                   3,
                                   {{"res", PT_ERRNO, PF_DEC},
                                    {"oldpath", PT_FSPATH, PF_NA},
                                    {"newpath", PT_FSPATH, PF_NA}}},
        [PPME_SYSCALL_RENAMEAT_E] = {"renameat", EC_FILE | EC_SYSCALL, EF_OLD_VERSION, 0},
        [PPME_SYSCALL_RENAMEAT_X] = {"renameat",
                                     EC_FILE | EC_SYSCALL,
                                     EF_NONE,
                                     5,
                                     {{"res", PT_ERRNO, PF_DEC},
                                      {"olddirfd", PT_FD, PF_DEC},
                                      {"oldpath", PT_FSRELPATH, PF_NA, DIRFD_PARAM(1)},
                                      {"newdirfd", PT_FD, PF_DEC},
                                      {"newpath", PT_FSRELPATH, PF_NA, DIRFD_PARAM(3)}}},
        [PPME_SYSCALL_SYMLINK_E] = {"symlink", EC_FILE | EC_SYSCALL, EF_OLD_VERSION, 0},
        [PPME_SYSCALL_SYMLINK_X] = {"symlink",
                                    EC_FILE | EC_SYSCALL,
                                    EF_NONE,
                                    3,
                                    {{"res", PT_ERRNO, PF_DEC},
                                     {"target", PT_CHARBUF, PF_NA},
                                     {"linkpath", PT_FSPATH, PF_NA}}},
        [PPME_SYSCALL_SYMLINKAT_E] = {"symlinkat", EC_FILE | EC_SYSCALL, EF_OLD_VERSION, 0},
        [PPME_SYSCALL_SYMLINKAT_X] = {"symlinkat",
                                      EC_FILE | EC_SYSCALL,
                                      EF_USES_FD,
                                      4,
                                      {{"res", PT_ERRNO, PF_DEC},
                                       {"target", PT_CHARBUF, PF_NA},
                                       {"linkdirfd", PT_FD, PF_DEC},
                                       {"linkpath", PT_FSRELPATH, PF_NA, DIRFD_PARAM(2)}}},
        [PPME_SYSCALL_FORK_E] = {"fork",
                                 EC_PROCESS | EC_SYSCALL,
                                 EF_OLD_VERSION | EF_MODIFIES_STATE,
                                 0},
        [PPME_SYSCALL_FORK_X] = {"fork",
                                 EC_PROCESS | EC_SYSCALL,
                                 EF_OLD_VERSION | EF_MODIFIES_STATE | EF_CONVERTER_MANAGED,
                                 16,
                                 {{"res", PT_PID, PF_DEC},
                                  {"exe", PT_CHARBUF, PF_NA},
                                  {"args", PT_BYTEBUF, PF_NA},
                                  {"tid", PT_PID, PF_DEC},
                                  {"pid", PT_PID, PF_DEC},
                                  {"ptid", PT_PID, PF_DEC},
                                  {"cwd", PT_CHARBUF, PF_NA},
                                  {"fdlimit", PT_INT64, PF_DEC},
                                  {"pgft_maj", PT_UINT64, PF_DEC},
                                  {"pgft_min", PT_UINT64, PF_DEC},
                                  {"vm_size", PT_UINT32, PF_DEC},
                                  {"vm_rss", PT_UINT32, PF_DEC},
                                  {"vm_swap", PT_UINT32, PF_DEC},
                                  {"flags", PT_FLAGS32, PF_HEX, clone_flags},
                                  {"uid", PT_UINT32, PF_DEC},
                                  {"gid", PT_UINT32, PF_DEC}}},
        [PPME_SYSCALL_VFORK_E] = {"vfork",
                                  EC_PROCESS | EC_SYSCALL,
                                  EF_OLD_VERSION | EF_MODIFIES_STATE,
                                  0},
        [PPME_SYSCALL_VFORK_X] = {"vfork",
                                  EC_PROCESS | EC_SYSCALL,
                                  EF_OLD_VERSION | EF_MODIFIES_STATE | EF_CONVERTER_MANAGED,
                                  16,
                                  {{"res", PT_PID, PF_DEC},
                                   {"exe", PT_CHARBUF, PF_NA},
                                   {"args", PT_BYTEBUF, PF_NA},
                                   {"tid", PT_PID, PF_DEC},
                                   {"pid", PT_PID, PF_DEC},
                                   {"ptid", PT_PID, PF_DEC},
                                   {"cwd", PT_CHARBUF, PF_NA},
                                   {"fdlimit", PT_INT64, PF_DEC},
                                   {"pgft_maj", PT_UINT64, PF_DEC},
                                   {"pgft_min", PT_UINT64, PF_DEC},
                                   {"vm_size", PT_UINT32, PF_DEC},
                                   {"vm_rss", PT_UINT32, PF_DEC},
                                   {"vm_swap", PT_UINT32, PF_DEC},
                                   {"flags", PT_FLAGS32, PF_HEX, clone_flags},
                                   {"uid", PT_UINT32, PF_DEC},
                                   {"gid", PT_UINT32, PF_DEC}}},
        [PPME_PROCEXIT_1_E] = {"procexit",
                               EC_PROCESS | EC_TRACEPOINT,
                               EF_MODIFIES_STATE,
                               5,
                               {{"status", PT_ERRNO, PF_DEC},
                                {"ret", PT_ERRNO, PF_DEC},
                                {"sig", PT_SIGTYPE, PF_DEC},
                                {"core", PT_UINT8, PF_DEC},
                                {"reaper_tid", PT_PID, PF_DEC}}},
        [PPME_PROCEXIT_1_X] = {"NA", EC_UNKNOWN, EF_UNUSED, 0},
        [PPME_SYSCALL_SENDFILE_E] = {"sendfile",
                                     EC_IO_WRITE | EC_SYSCALL,
                                     EF_OLD_VERSION | EF_USES_FD | EF_CONVERTER_MANAGED,
                                     4,
                                     {{"out_fd", PT_FD, PF_DEC},
                                      {"in_fd", PT_FD, PF_DEC},
                                      {"offset", PT_UINT64, PF_DEC},
                                      {"size", PT_UINT64, PF_DEC}}},
        [PPME_SYSCALL_SENDFILE_X] = {"sendfile",
                                     EC_IO_WRITE | EC_SYSCALL,
                                     EF_USES_FD | EF_CONVERTER_MANAGED,
                                     5,
                                     {{"res", PT_ERRNO, PF_DEC},
                                      {"offset", PT_UINT64, PF_DEC},
                                      {"out_fd", PT_FD, PF_DEC},
                                      {"in_fd", PT_FD, PF_DEC},
                                      {"size", PT_UINT64, PF_DEC}}},
        [PPME_SYSCALL_QUOTACTL_E] = {"quotactl",
                                     EC_USER | EC_SYSCALL,
                                     EF_OLD_VERSION | EF_CONVERTER_MANAGED,
                                     4,
                                     {{"cmd", PT_FLAGS16, PF_DEC, quotactl_cmds},
                                      {"type", PT_FLAGS8, PF_DEC, quotactl_types},
                                      {"id", PT_UINT32, PF_DEC},
                                      {"quota_fmt", PT_FLAGS8, PF_DEC, quotactl_quota_fmts}}},
        [PPME_SYSCALL_QUOTACTL_X] = {"quotactl",
                                     EC_USER | EC_SYSCALL,
                                     EF_CONVERTER_MANAGED,
                                     18,
                                     {{"res", PT_ERRNO, PF_DEC},
                                      {"special", PT_CHARBUF, PF_NA},
                                      {"quotafilepath", PT_CHARBUF, PF_NA},
                                      {"dqb_bhardlimit", PT_UINT64, PF_DEC},
                                      {"dqb_bsoftlimit", PT_UINT64, PF_DEC},
                                      {"dqb_curspace", PT_UINT64, PF_DEC},
                                      {"dqb_ihardlimit", PT_UINT64, PF_DEC},
                                      {"dqb_isoftlimit", PT_UINT64, PF_DEC},
                                      {"dqb_btime", PT_RELTIME, PF_DEC},
                                      {"dqb_itime", PT_RELTIME, PF_DEC},
                                      {"dqi_bgrace", PT_RELTIME, PF_DEC},
                                      {"dqi_igrace", PT_RELTIME, PF_DEC},
                                      {"dqi_flags", PT_FLAGS8, PF_DEC, quotactl_dqi_flags},
                                      {"quota_fmt_out", PT_FLAGS8, PF_DEC, quotactl_quota_fmts},
                                      {"cmd", PT_FLAGS16, PF_DEC, quotactl_cmds},
                                      {"type", PT_FLAGS8, PF_DEC, quotactl_types},
                                      {"id", PT_UINT32, PF_DEC},
                                      {"quota_fmt", PT_FLAGS8, PF_DEC, quotactl_quota_fmts}}},
        [PPME_SYSCALL_SETRESUID_E] = {"setresuid",
                                      EC_USER | EC_SYSCALL,
                                      EF_OLD_VERSION | EF_MODIFIES_STATE | EF_CONVERTER_MANAGED,
                                      3,
                                      {{"ruid", PT_UID, PF_DEC},
                                       {"euid", PT_UID, PF_DEC},
                                       {"suid", PT_UID, PF_DEC}}},
        [PPME_SYSCALL_SETRESUID_X] = {"setresuid",
                                      EC_USER | EC_SYSCALL,
                                      EF_MODIFIES_STATE | EF_CONVERTER_MANAGED,
                                      4,
                                      {{"res", PT_ERRNO, PF_DEC},
                                       {"ruid", PT_UID, PF_DEC},
                                       {"euid", PT_UID, PF_DEC},
                                       {"suid", PT_UID, PF_DEC}}},
        [PPME_SYSCALL_SETRESGID_E] = {"setresgid",
                                      EC_USER | EC_SYSCALL,
                                      EF_OLD_VERSION | EF_MODIFIES_STATE | EF_CONVERTER_MANAGED,
                                      3,
                                      {{"rgid", PT_GID, PF_DEC},
                                       {"egid", PT_GID, PF_DEC},
                                       {"sgid", PT_GID, PF_DEC}}},
        [PPME_SYSCALL_SETRESGID_X] = {"setresgid",
                                      EC_USER | EC_SYSCALL,
                                      EF_MODIFIES_STATE | EF_CONVERTER_MANAGED,
                                      4,
                                      {{"res", PT_ERRNO, PF_DEC},
                                       {"rgid", PT_GID, PF_DEC},
                                       {"egid", PT_GID, PF_DEC},
                                       {"sgid", PT_GID, PF_DEC}}},
        [PPME_SCAPEVENT_E] = {"scapevent",
                              EC_INTERNAL | EC_METAEVENT,
                              EF_SKIPPARSERESET,
                              2,
                              {{"event_type", PT_UINT32, PF_DEC},
                               {"event_data", PT_UINT64, PF_DEC}}},
        [PPME_SCAPEVENT_X] = {"NA", EC_UNKNOWN, EF_UNUSED, 0},
        [PPME_SYSCALL_SETUID_E] = {"setuid",
                                   EC_USER | EC_SYSCALL,
                                   EF_OLD_VERSION | EF_MODIFIES_STATE | EF_CONVERTER_MANAGED,
                                   1,
                                   {{"uid", PT_UID, PF_DEC}}},
        [PPME_SYSCALL_SETUID_X] = {"setuid",
                                   EC_USER | EC_SYSCALL,
                                   EF_MODIFIES_STATE | EF_CONVERTER_MANAGED,
                                   2,
                                   {{"res", PT_ERRNO, PF_DEC}, {"uid", PT_UID, PF_DEC}}},
        [PPME_SYSCALL_SETGID_E] = {"setgid",
                                   EC_USER | EC_SYSCALL,
                                   EF_OLD_VERSION | EF_MODIFIES_STATE | EF_CONVERTER_MANAGED,
                                   1,
                                   {{"gid", PT_GID, PF_DEC}}},
        [PPME_SYSCALL_SETGID_X] = {"setgid",
                                   EC_USER | EC_SYSCALL,
                                   EF_MODIFIES_STATE | EF_CONVERTER_MANAGED,
                                   2,
                                   {{"res", PT_ERRNO, PF_DEC}, {"gid", PT_GID, PF_DEC}}},
        [PPME_SYSCALL_GETUID_E] = {"getuid", EC_USER | EC_SYSCALL, EF_OLD_VERSION, 0},
        [PPME_SYSCALL_GETUID_X] =
                {"getuid", EC_USER | EC_SYSCALL, EF_NONE, 1, {{"uid", PT_UID, PF_DEC}}},
        [PPME_SYSCALL_GETEUID_E] = {"geteuid", EC_USER | EC_SYSCALL, EF_OLD_VERSION, 0},
        [PPME_SYSCALL_GETEUID_X] =
                {"geteuid", EC_USER | EC_SYSCALL, EF_NONE, 1, {{"euid", PT_UID, PF_DEC}}},
        [PPME_SYSCALL_GETGID_E] = {"getgid", EC_USER | EC_SYSCALL, EF_OLD_VERSION, 0},
        [PPME_SYSCALL_GETGID_X] =
                {"getgid", EC_USER | EC_SYSCALL, EF_NONE, 1, {{"gid", PT_GID, PF_DEC}}},
        [PPME_SYSCALL_GETEGID_E] = {"getegid", EC_USER | EC_SYSCALL, EF_OLD_VERSION, 0},
        [PPME_SYSCALL_GETEGID_X] =
                {"getegid", EC_USER | EC_SYSCALL, EF_NONE, 1, {{"egid", PT_GID, PF_DEC}}},
        [PPME_SYSCALL_GETRESUID_E] = {"getresuid", EC_USER | EC_SYSCALL, EF_OLD_VERSION, 0},
        [PPME_SYSCALL_GETRESUID_X] = {"getresuid",
                                      EC_USER | EC_SYSCALL,
                                      EF_NONE,
                                      4,
                                      {{"res", PT_ERRNO, PF_DEC},
                                       {"ruid", PT_UID, PF_DEC},
                                       {"euid", PT_UID, PF_DEC},
                                       {"suid", PT_UID, PF_DEC}}},
        [PPME_SYSCALL_GETRESGID_E] = {"getresgid", EC_USER | EC_SYSCALL, EF_OLD_VERSION, 0},
        [PPME_SYSCALL_GETRESGID_X] = {"getresgid",
                                      EC_USER | EC_SYSCALL,
                                      EF_NONE,
                                      4,
                                      {{"res", PT_ERRNO, PF_DEC},
                                       {"rgid", PT_GID, PF_DEC},
                                       {"egid", PT_GID, PF_DEC},
                                       {"sgid", PT_GID, PF_DEC}}},
        [PPME_SYSCALL_EXECVE_15_E] = {"execve",
                                      EC_PROCESS | EC_SYSCALL,
                                      EF_OLD_VERSION | EF_MODIFIES_STATE | EF_CONVERTER_MANAGED,
                                      0},
        [PPME_SYSCALL_EXECVE_15_X] = {"execve",
                                      EC_PROCESS | EC_SYSCALL,
                                      EF_OLD_VERSION | EF_MODIFIES_STATE | EF_CONVERTER_MANAGED,
                                      15,
                                      {{"res", PT_ERRNO, PF_DEC},
                                       {"exe", PT_CHARBUF, PF_NA},
                                       {"args", PT_BYTEBUF, PF_NA},
                                       {"tid", PT_PID, PF_DEC},
                                       {"pid", PT_PID, PF_DEC},
                                       {"ptid", PT_PID, PF_DEC},
                                       {"cwd", PT_CHARBUF, PF_NA},
                                       {"fdlimit", PT_UINT64, PF_DEC},
                                       {"pgft_maj", PT_UINT64, PF_DEC},
                                       {"pgft_min", PT_UINT64, PF_DEC},
                                       {"vm_size", PT_UINT32, PF_DEC},
                                       {"vm_rss", PT_UINT32, PF_DEC},
                                       {"vm_swap", PT_UINT32, PF_DEC},
                                       {"comm", PT_CHARBUF, PF_NA},
                                       {"env", PT_BYTEBUF, PF_NA}}},
        [PPME_SYSCALL_CLONE_17_E] = {"clone",
                                     EC_PROCESS | EC_SYSCALL,
                                     EF_OLD_VERSION | EF_MODIFIES_STATE,
                                     0},
        [PPME_SYSCALL_CLONE_17_X] = {"clone",
                                     EC_PROCESS | EC_SYSCALL,
                                     EF_OLD_VERSION | EF_MODIFIES_STATE | EF_CONVERTER_MANAGED,
                                     17,
                                     {{"res", PT_PID, PF_DEC},
                                      {"exe", PT_CHARBUF, PF_NA},
                                      {"args", PT_BYTEBUF, PF_NA},
                                      {"tid", PT_PID, PF_DEC},
                                      {"pid", PT_PID, PF_DEC},
                                      {"ptid", PT_PID, PF_DEC},
                                      {"cwd", PT_CHARBUF, PF_NA},
                                      {"fdlimit", PT_INT64, PF_DEC},
                                      {"pgft_maj", PT_UINT64, PF_DEC},
                                      {"pgft_min", PT_UINT64, PF_DEC},
                                      {"vm_size", PT_UINT32, PF_DEC},
                                      {"vm_rss", PT_UINT32, PF_DEC},
                                      {"vm_swap", PT_UINT32, PF_DEC},
                                      {"comm", PT_CHARBUF, PF_NA},
                                      {"flags", PT_FLAGS32, PF_HEX, clone_flags},
                                      {"uid", PT_UINT32, PF_DEC},
                                      {"gid", PT_UINT32, PF_DEC}}},
        [PPME_SYSCALL_FORK_17_E] = {"fork",
                                    EC_PROCESS | EC_SYSCALL,
                                    EF_OLD_VERSION | EF_MODIFIES_STATE,
                                    0},
        [PPME_SYSCALL_FORK_17_X] = {"fork",
                                    EC_PROCESS | EC_SYSCALL,
                                    EF_OLD_VERSION | EF_MODIFIES_STATE | EF_CONVERTER_MANAGED,
                                    17,
                                    {{"res", PT_PID, PF_DEC},
                                     {"exe", PT_CHARBUF, PF_NA},
                                     {"args", PT_BYTEBUF, PF_NA},
                                     {"tid", PT_PID, PF_DEC},
                                     {"pid", PT_PID, PF_DEC},
                                     {"ptid", PT_PID, PF_DEC},
                                     {"cwd", PT_CHARBUF, PF_NA},
                                     {"fdlimit", PT_INT64, PF_DEC},
                                     {"pgft_maj", PT_UINT64, PF_DEC},
                                     {"pgft_min", PT_UINT64, PF_DEC},
                                     {"vm_size", PT_UINT32, PF_DEC},
                                     {"vm_rss", PT_UINT32, PF_DEC},
                                     {"vm_swap", PT_UINT32, PF_DEC},
                                     {"comm", PT_CHARBUF, PF_NA},
                                     {"flags", PT_FLAGS32, PF_HEX, clone_flags},
                                     {"uid", PT_UINT32, PF_DEC},
                                     {"gid", PT_UINT32, PF_DEC}}},
        [PPME_SYSCALL_VFORK_17_E] = {"vfork",
                                     EC_PROCESS | EC_SYSCALL,
                                     EF_OLD_VERSION | EF_MODIFIES_STATE,
                                     0},
        [PPME_SYSCALL_VFORK_17_X] = {"vfork",
                                     EC_PROCESS | EC_SYSCALL,
                                     EF_OLD_VERSION | EF_MODIFIES_STATE | EF_CONVERTER_MANAGED,
                                     17,
                                     {{"res", PT_PID, PF_DEC},
                                      {"exe", PT_CHARBUF, PF_NA},
                                      {"args", PT_BYTEBUF, PF_NA},
                                      {"tid", PT_PID, PF_DEC},
                                      {"pid", PT_PID, PF_DEC},
                                      {"ptid", PT_PID, PF_DEC},
                                      {"cwd", PT_CHARBUF, PF_NA},
                                      {"fdlimit", PT_INT64, PF_DEC},
                                      {"pgft_maj", PT_UINT64, PF_DEC},
                                      {"pgft_min", PT_UINT64, PF_DEC},
                                      {"vm_size", PT_UINT32, PF_DEC},
                                      {"vm_rss", PT_UINT32, PF_DEC},
                                      {"vm_swap", PT_UINT32, PF_DEC},
                                      {"comm", PT_CHARBUF, PF_NA},
                                      {"flags", PT_FLAGS32, PF_HEX, clone_flags},
                                      {"uid", PT_UINT32, PF_DEC},
                                      {"gid", PT_UINT32, PF_DEC}}},
        [PPME_SYSCALL_CLONE_20_E] = {"clone",
                                     EC_PROCESS | EC_SYSCALL,
                                     EF_OLD_VERSION | EF_MODIFIES_STATE,
                                     0},
        [PPME_SYSCALL_CLONE_20_X] = {"clone",
                                     EC_PROCESS | EC_SYSCALL,
                                     EF_MODIFIES_STATE | EF_CONVERTER_MANAGED,
                                     21,
                                     {{"res", PT_PID, PF_DEC},
                                      {"exe", PT_CHARBUF, PF_NA},
                                      {"args", PT_BYTEBUF, PF_NA},
                                      {"tid", PT_PID, PF_DEC},
                                      {"pid", PT_PID, PF_DEC},
                                      {"ptid", PT_PID, PF_DEC},
                                      {"cwd", PT_CHARBUF, PF_NA},
                                      {"fdlimit", PT_INT64, PF_DEC},
                                      {"pgft_maj", PT_UINT64, PF_DEC},
                                      {"pgft_min", PT_UINT64, PF_DEC},
                                      {"vm_size", PT_UINT32, PF_DEC},
                                      {"vm_rss", PT_UINT32, PF_DEC},
                                      {"vm_swap", PT_UINT32, PF_DEC},
                                      {"comm", PT_CHARBUF, PF_NA},
                                      {"cgroups", PT_BYTEBUF, PF_NA},
                                      {"flags", PT_FLAGS32, PF_HEX, clone_flags},
                                      {"uid", PT_UINT32, PF_DEC},
                                      {"gid", PT_UINT32, PF_DEC},
                                      {"vtid", PT_PID, PF_DEC},
                                      {"vpid", PT_PID, PF_DEC},
                                      {"pidns_init_start_ts", PT_UINT64, PF_DEC}}},
        [PPME_SYSCALL_FORK_20_E] = {"fork",
                                    EC_PROCESS | EC_SYSCALL,
                                    EF_OLD_VERSION | EF_MODIFIES_STATE,
                                    0},
        [PPME_SYSCALL_FORK_20_X] = {"fork",
                                    EC_PROCESS | EC_SYSCALL,
                                    EF_MODIFIES_STATE | EF_CONVERTER_MANAGED,
                                    21,
                                    {{"res", PT_PID, PF_DEC},
                                     {"exe", PT_CHARBUF, PF_NA},
                                     {"args", PT_BYTEBUF, PF_NA},
                                     {"tid", PT_PID, PF_DEC},
                                     {"pid", PT_PID, PF_DEC},
                                     {"ptid", PT_PID, PF_DEC},
                                     {"cwd", PT_CHARBUF, PF_NA},
                                     {"fdlimit", PT_INT64, PF_DEC},
                                     {"pgft_maj", PT_UINT64, PF_DEC},
                                     {"pgft_min", PT_UINT64, PF_DEC},
                                     {"vm_size", PT_UINT32, PF_DEC},
                                     {"vm_rss", PT_UINT32, PF_DEC},
                                     {"vm_swap", PT_UINT32, PF_DEC},
                                     {"comm", PT_CHARBUF, PF_NA},
                                     {"cgroups", PT_BYTEBUF, PF_NA},
                                     {"flags", PT_FLAGS32, PF_HEX, clone_flags},
                                     {"uid", PT_UINT32, PF_DEC},
                                     {"gid", PT_UINT32, PF_DEC},
                                     {"vtid", PT_PID, PF_DEC},
                                     {"vpid", PT_PID, PF_DEC},
                                     {"pidns_init_start_ts", PT_UINT64, PF_DEC}}},
        [PPME_SYSCALL_VFORK_20_E] = {"vfork",
                                     EC_PROCESS | EC_SYSCALL,
                                     EF_OLD_VERSION | EF_MODIFIES_STATE,
                                     0},
        [PPME_SYSCALL_VFORK_20_X] = {"vfork",
                                     EC_PROCESS | EC_SYSCALL,
                                     EF_MODIFIES_STATE | EF_CONVERTER_MANAGED,
                                     21,
                                     {{"res", PT_PID, PF_DEC},
                                      {"exe", PT_CHARBUF, PF_NA},
                                      {"args", PT_BYTEBUF, PF_NA},
                                      {"tid", PT_PID, PF_DEC},
                                      {"pid", PT_PID, PF_DEC},
                                      {"ptid", PT_PID, PF_DEC},
                                      {"cwd", PT_CHARBUF, PF_NA},
                                      {"fdlimit", PT_INT64, PF_DEC},
                                      {"pgft_maj", PT_UINT64, PF_DEC},
                                      {"pgft_min", PT_UINT64, PF_DEC},
                                      {"vm_size", PT_UINT32, PF_DEC},
                                      {"vm_rss", PT_UINT32, PF_DEC},
                                      {"vm_swap", PT_UINT32, PF_DEC},
                                      {"comm", PT_CHARBUF, PF_NA},
                                      {"cgroups", PT_BYTEBUF, PF_NA},
                                      {"flags", PT_FLAGS32, PF_HEX, clone_flags},
                                      {"uid", PT_UINT32, PF_DEC},
                                      {"gid", PT_UINT32, PF_DEC},
                                      {"vtid", PT_PID, PF_DEC},
                                      {"vpid", PT_PID, PF_DEC},
                                      {"pidns_init_start_ts", PT_UINT64, PF_DEC}}},
        [PPME_CONTAINER_E] = {"container",
                              EC_INTERNAL | EC_METAEVENT,
                              EF_OLD_VERSION | EF_SKIPPARSERESET | EF_MODIFIES_STATE |
                                      EF_CONVERTER_MANAGED,
                              4,
                              {{"id", PT_CHARBUF, PF_NA},
                               {"type", PT_UINT32, PF_DEC},
                               {"name", PT_CHARBUF, PF_NA},
                               {"image", PT_CHARBUF, PF_NA}}},
        [PPME_CONTAINER_X] = {"NA", EC_UNKNOWN, EF_UNUSED, 0},
        [PPME_SYSCALL_EXECVE_16_E] = {"execve",
                                      EC_PROCESS | EC_SYSCALL,
                                      EF_OLD_VERSION | EF_MODIFIES_STATE | EF_CONVERTER_MANAGED,
                                      0},
        [PPME_SYSCALL_EXECVE_16_X] = {"execve",
                                      EC_PROCESS | EC_SYSCALL,
                                      EF_OLD_VERSION | EF_MODIFIES_STATE | EF_CONVERTER_MANAGED,
                                      16,
                                      {{"res", PT_ERRNO, PF_DEC},
                                       {"exe", PT_CHARBUF, PF_NA},
                                       {"args", PT_BYTEBUF, PF_NA},
                                       {"tid", PT_PID, PF_DEC},
                                       {"pid", PT_PID, PF_DEC},
                                       {"ptid", PT_PID, PF_DEC},
                                       {"cwd", PT_CHARBUF, PF_NA},
                                       {"fdlimit", PT_UINT64, PF_DEC},
                                       {"pgft_maj", PT_UINT64, PF_DEC},
                                       {"pgft_min", PT_UINT64, PF_DEC},
                                       {"vm_size", PT_UINT32, PF_DEC},
                                       {"vm_rss", PT_UINT32, PF_DEC},
                                       {"vm_swap", PT_UINT32, PF_DEC},
                                       {"comm", PT_CHARBUF, PF_NA},
                                       {"cgroups", PT_BYTEBUF, PF_NA},
                                       {"env", PT_BYTEBUF, PF_NA}}},
        [PPME_SIGNALDELIVER_E] = {"signaldeliver",
                                  EC_SIGNAL | EC_TRACEPOINT,
                                  EF_NONE,
                                  3,
                                  {{"spid", PT_PID, PF_DEC},
                                   {"dpid", PT_PID, PF_DEC},
                                   {"sig", PT_SIGTYPE, PF_DEC}}},
        [PPME_SIGNALDELIVER_X] = {"NA", EC_UNKNOWN, EF_UNUSED, 0},
        [PPME_PROCINFO_E] = {"procinfo",
                             EC_INTERNAL | EC_METAEVENT,
                             EF_SKIPPARSERESET,
                             2,
                             {{"cpu_usr", PT_UINT64, PF_DEC}, {"cpu_sys", PT_UINT64, PF_DEC}}},
        [PPME_PROCINFO_X] = {"NA", EC_UNKNOWN, EF_UNUSED, 0},
        [PPME_SYSCALL_GETDENTS_E] = {"getdents",
                                     EC_FILE | EC_SYSCALL,
                                     EF_OLD_VERSION | EF_USES_FD | EF_CONVERTER_MANAGED,
                                     1,
                                     {{"fd", PT_FD, PF_NA}}},
        [PPME_SYSCALL_GETDENTS_X] = {"getdents",
                                     EC_FILE | EC_SYSCALL,
                                     EF_USES_FD | EF_CONVERTER_MANAGED,
                                     2,
                                     {{"res", PT_ERRNO, PF_DEC}, {"fd", PT_FD, PF_NA}}},
        [PPME_SYSCALL_GETDENTS64_E] = {"getdents64",
                                       EC_FILE | EC_SYSCALL,
                                       EF_OLD_VERSION | EF_USES_FD | EF_CONVERTER_MANAGED,
                                       1,
                                       {{"fd", PT_FD, PF_NA}}},
        [PPME_SYSCALL_GETDENTS64_X] = {"getdents64",
                                       EC_FILE | EC_SYSCALL,
                                       EF_USES_FD | EF_CONVERTER_MANAGED,
                                       2,
                                       {{"res", PT_ERRNO, PF_DEC}, {"fd", PT_FD, PF_NA}}},
        [PPME_SYSCALL_SETNS_E] = {"setns",
                                  EC_PROCESS | EC_SYSCALL,
                                  EF_OLD_VERSION | EF_USES_FD | EF_CONVERTER_MANAGED,
                                  2,
                                  {{"fd", PT_FD, PF_NA},
                                   {"nstype", PT_FLAGS32, PF_HEX, clone_flags}}},
        [PPME_SYSCALL_SETNS_X] = {"setns",
                                  EC_PROCESS | EC_SYSCALL,
                                  EF_USES_FD | EF_CONVERTER_MANAGED,
                                  3,
                                  {{"res", PT_ERRNO, PF_DEC},
                                   {"fd", PT_FD, PF_NA},
                                   {"nstype", PT_FLAGS32, PF_HEX, clone_flags}}},
        [PPME_SYSCALL_FLOCK_E] = {"flock",
                                  EC_FILE | EC_SYSCALL,
                                  EF_OLD_VERSION | EF_USES_FD | EF_CONVERTER_MANAGED,
                                  2,
                                  {{"fd", PT_FD, PF_NA},
                                   {"operation", PT_FLAGS32, PF_HEX, flock_flags}}},
        [PPME_SYSCALL_FLOCK_X] = {"flock",
                                  EC_FILE | EC_SYSCALL,
                                  EF_USES_FD | EF_CONVERTER_MANAGED,
                                  3,
                                  {{"res", PT_ERRNO, PF_DEC},
                                   {"fd", PT_FD, PF_NA},
                                   {"operation", PT_FLAGS32, PF_HEX, flock_flags}}},
        [PPME_CPU_HOTPLUG_E] = {"cpu_hotplug",
                                EC_SYSTEM | EC_METAEVENT,
                                EF_SKIPPARSERESET | EF_MODIFIES_STATE,
                                2,
                                {{"cpu", PT_UINT32, PF_DEC}, {"action", PT_UINT32, PF_DEC}}},
        [PPME_CPU_HOTPLUG_X] = {"NA", EC_UNKNOWN, EF_UNUSED, 0},
        [PPME_SOCKET_ACCEPT_5_E] = {"accept",
                                    EC_NET | EC_SYSCALL,
                                    EF_OLD_VERSION | EF_CREATES_FD | EF_MODIFIES_STATE,
                                    0},
        [PPME_SOCKET_ACCEPT_5_X] = {"accept",
                                    EC_NET | EC_SYSCALL,
                                    EF_CREATES_FD | EF_MODIFIES_STATE,
                                    5,
                                    {{"fd", PT_FD, PF_DEC},
                                     {"tuple", PT_SOCKTUPLE, PF_NA},
                                     {"queuepct", PT_UINT8, PF_DEC},
                                     {"queuelen", PT_UINT32, PF_DEC},
                                     {"queuemax", PT_UINT32, PF_DEC}}},
        [PPME_SOCKET_ACCEPT4_5_E] = {"accept",
                                     EC_NET | EC_SYSCALL,
                                     EF_OLD_VERSION | EF_CREATES_FD | EF_MODIFIES_STATE |
                                             EF_CONVERTER_MANAGED,
                                     1,
                                     {{"flags", PT_INT32, PF_HEX}}},
        [PPME_SOCKET_ACCEPT4_5_X] = {"accept",
                                     EC_NET | EC_SYSCALL,
                                     EF_OLD_VERSION | EF_CREATES_FD | EF_MODIFIES_STATE |
                                             EF_CONVERTER_MANAGED,
                                     5,
                                     {{"fd", PT_FD, PF_DEC},
                                      {"tuple", PT_SOCKTUPLE, PF_NA},
                                      {"queuepct", PT_UINT8, PF_DEC},
                                      {"queuelen", PT_UINT32, PF_DEC},
                                      {"queuemax", PT_UINT32, PF_DEC}}},
        [PPME_SYSCALL_SEMOP_E] = {"semop",
                                  EC_PROCESS | EC_SYSCALL,
                                  EF_OLD_VERSION | EF_CONVERTER_MANAGED,
                                  1,
                                  {{"semid", PT_INT32, PF_DEC}}},
        [PPME_SYSCALL_SEMOP_X] = {"semop",
                                  EC_PROCESS | EC_SYSCALL,
                                  EF_CONVERTER_MANAGED,
                                  9,
                                  {{"res", PT_ERRNO, PF_DEC},
                                   {"nsops", PT_UINT32, PF_DEC},
                                   {"sem_num_0", PT_UINT16, PF_DEC},
                                   {"sem_op_0", PT_INT16, PF_DEC},
                                   {"sem_flg_0", PT_FLAGS16, PF_HEX, semop_flags},
                                   {"sem_num_1", PT_UINT16, PF_DEC},
                                   {"sem_op_1", PT_INT16, PF_DEC},
                                   {"sem_flg_1", PT_FLAGS16, PF_HEX, semop_flags},
                                   {"semid", PT_INT32, PF_DEC}}},
        [PPME_SYSCALL_SEMCTL_E] = {"semctl",
                                   EC_PROCESS | EC_SYSCALL,
                                   EF_OLD_VERSION | EF_CONVERTER_MANAGED,
                                   4,
                                   {{"semid", PT_INT32, PF_DEC},
                                    {"semnum", PT_INT32, PF_DEC},
                                    {"cmd", PT_FLAGS16, PF_HEX, semctl_commands},
                                    {"val", PT_INT32, PF_DEC}}},
        [PPME_SYSCALL_SEMCTL_X] = {"semctl",
                                   EC_PROCESS | EC_SYSCALL,
                                   EF_CONVERTER_MANAGED,
                                   5,
                                   {{"res", PT_ERRNO, PF_DEC},
                                    {"semid", PT_INT32, PF_DEC},
                                    {"semnum", PT_INT32, PF_DEC},
                                    {"cmd", PT_FLAGS16, PF_HEX, semctl_commands},
                                    {"val", PT_INT32, PF_DEC}}},
        [PPME_SYSCALL_PPOLL_E] = {"ppoll",
                                  EC_WAIT | EC_SYSCALL,
                                  EF_OLD_VERSION | EF_WAITS | EF_CONVERTER_MANAGED,
                                  3,
                                  {{"fds", PT_FDLIST, PF_DEC},
                                   {"timeout", PT_RELTIME, PF_DEC},
                                   {"sigmask", PT_SIGSET, PF_DEC}}},
        [PPME_SYSCALL_PPOLL_X] = {"ppoll",
                                  EC_WAIT | EC_SYSCALL,
                                  EF_WAITS | EF_CONVERTER_MANAGED,
                                  4,
                                  {{"res", PT_ERRNO, PF_DEC},
                                   {"fds", PT_FDLIST, PF_DEC},
                                   {"timeout", PT_RELTIME, PF_DEC},
                                   {"sigmask", PT_SIGSET, PF_DEC}}},
        [PPME_SYSCALL_MOUNT_E] = {"mount",
                                  EC_FILE | EC_SYSCALL,
                                  EF_OLD_VERSION | EF_MODIFIES_STATE | EF_CONVERTER_MANAGED,
                                  1,
                                  {{"flags", PT_FLAGS32, PF_HEX, mount_flags}}},
        [PPME_SYSCALL_MOUNT_X] = {"mount",
                                  EC_FILE | EC_SYSCALL,
                                  EF_MODIFIES_STATE | EF_CONVERTER_MANAGED,
                                  5,
                                  {{"res", PT_ERRNO, PF_DEC},
                                   {"dev", PT_CHARBUF, PF_NA},
                                   {"dir", PT_FSPATH, PF_NA},
                                   {"type", PT_CHARBUF, PF_NA},
                                   {"flags", PT_FLAGS32, PF_HEX, mount_flags}}},
        [PPME_SYSCALL_UMOUNT_E] = {"umount",
                                   EC_FILE | EC_SYSCALL,
                                   EF_OLD_VERSION | EF_MODIFIES_STATE,
                                   1,
                                   {{"flags", PT_FLAGS32, PF_HEX, umount_flags}}},
        [PPME_SYSCALL_UMOUNT_X] = {"umount",
                                   EC_FILE | EC_SYSCALL,
                                   EF_OLD_VERSION | EF_MODIFIES_STATE | EF_CONVERTER_MANAGED,
                                   2,
                                   {{"res", PT_ERRNO, PF_DEC}, {"name", PT_FSPATH, PF_NA}}},
        [PPME_K8S_E] = {"k8s",
                        EC_INTERNAL | EC_METAEVENT,
                        EF_OLD_VERSION | EF_SKIPPARSERESET | EF_MODIFIES_STATE,
                        1,
                        {{"json", PT_CHARBUF, PF_NA}}},
        [PPME_K8S_X] = {"NA", EC_UNKNOWN, EF_UNUSED, 0},
        [PPME_SYSCALL_SEMGET_E] = {"semget",
                                   EC_PROCESS | EC_SYSCALL,
                                   EF_OLD_VERSION | EF_CONVERTER_MANAGED,
                                   3,
                                   {{"key", PT_INT32, PF_HEX},
                                    {"nsems", PT_INT32, PF_DEC},
                                    {"semflg", PT_FLAGS32, PF_HEX, semget_flags}}},
        [PPME_SYSCALL_SEMGET_X] = {"semget",
                                   EC_PROCESS | EC_SYSCALL,
                                   EF_CONVERTER_MANAGED,
                                   4,
                                   {{"res", PT_ERRNO, PF_DEC},
                                    {"key", PT_INT32, PF_HEX},
                                    {"nsems", PT_INT32, PF_DEC},
                                    {"semflg", PT_FLAGS32, PF_HEX, semget_flags}}},
        [PPME_SYSCALL_ACCESS_E] = {"access",
                                   EC_FILE | EC_SYSCALL,
                                   EF_OLD_VERSION | EF_CONVERTER_MANAGED,
                                   1,
                                   {{"mode", PT_FLAGS32, PF_HEX, access_flags}}},
        [PPME_SYSCALL_ACCESS_X] = {"access",
                                   EC_FILE | EC_SYSCALL,
                                   EF_CONVERTER_MANAGED,
                                   3,
                                   {{"res", PT_ERRNO, PF_DEC},
                                    {"name", PT_FSPATH, PF_NA},
                                    {"mode", PT_FLAGS32, PF_HEX, access_flags}}},
        [PPME_SYSCALL_CHROOT_E] = {"chroot",
                                   EC_PROCESS | EC_SYSCALL,
                                   EF_OLD_VERSION | EF_MODIFIES_STATE,
                                   0},
        [PPME_SYSCALL_CHROOT_X] = {"chroot",
                                   EC_PROCESS | EC_SYSCALL,
                                   EF_MODIFIES_STATE,
                                   2,
                                   {{"res", PT_ERRNO, PF_DEC}, {"path", PT_FSPATH, PF_NA}}},
        [PPME_TRACER_E] = {"tracer",
                           EC_OTHER | EC_METAEVENT,
                           EF_OLD_VERSION,
                           3,
                           {{"id", PT_INT64, PF_DEC},
                            {"tags", PT_CHARBUFARRAY, PF_NA},
                            {"args", PT_CHARBUF_PAIR_ARRAY, PF_NA}}},
        [PPME_TRACER_X] = {"NA",
                           EC_UNKNOWN,
                           EF_UNUSED,
                           3,
                           {{"id", PT_INT64, PF_DEC},
                            {"tags", PT_CHARBUFARRAY, PF_NA},
                            {"args", PT_CHARBUF_PAIR_ARRAY, PF_NA}}},
        [PPME_MESOS_E] = {"mesos",
                          EC_INTERNAL | EC_METAEVENT,
                          EF_OLD_VERSION | EF_SKIPPARSERESET | EF_MODIFIES_STATE,
                          1,
                          {{"json", PT_CHARBUF, PF_NA}}},
        [PPME_MESOS_X] = {"NA", EC_UNKNOWN, EF_UNUSED, 0},
        [PPME_CONTAINER_JSON_E] =
                {"container",
                 EC_PROCESS | EC_METAEVENT,
                 EF_OLD_VERSION | EF_MODIFIES_STATE | EF_CONVERTER_MANAGED,
                 1,
                 {{"json", PT_CHARBUF, PF_NA}}},  /// TODO: do we need SKIPPARSERESET flag?
        [PPME_CONTAINER_JSON_X] = {"NA", EC_UNKNOWN, EF_UNUSED, 0},
        [PPME_SYSCALL_SETSID_E] = {"setsid",
                                   EC_PROCESS | EC_SYSCALL,
                                   EF_OLD_VERSION | EF_MODIFIES_STATE,
                                   0},
        [PPME_SYSCALL_SETSID_X] = {"setsid",
                                   EC_PROCESS | EC_SYSCALL,
                                   EF_MODIFIES_STATE,
                                   1,
                                   {{"res", PT_PID, PF_DEC}}},
        [PPME_SYSCALL_MKDIR_2_E] = {"mkdir",
                                    EC_FILE | EC_SYSCALL,
                                    EF_OLD_VERSION | EF_CONVERTER_MANAGED,
                                    1,
                                    {{"mode", PT_UINT32, PF_HEX}}},
        [PPME_SYSCALL_MKDIR_2_X] = {"mkdir",
                                    EC_FILE | EC_SYSCALL,
                                    EF_CONVERTER_MANAGED,
                                    3,
                                    {{"res", PT_ERRNO, PF_DEC},
                                     {"path", PT_FSPATH, PF_NA},
                                     {"mode", PT_UINT32, PF_HEX}}},
        [PPME_SYSCALL_RMDIR_2_E] = {"rmdir", EC_FILE | EC_SYSCALL, EF_OLD_VERSION, 0},
        [PPME_SYSCALL_RMDIR_2_X] = {"rmdir",
                                    EC_FILE | EC_SYSCALL,
                                    EF_NONE,
                                    2,
                                    {{"res", PT_ERRNO, PF_DEC}, {"path", PT_FSPATH, PF_NA}}},
        [PPME_NOTIFICATION_E] = {"notification",
                                 EC_OTHER | EC_METAEVENT,
                                 EF_SKIPPARSERESET,
                                 2,
                                 {
                                         {"id", PT_CHARBUF, PF_DEC},
                                         {"desc", PT_CHARBUF, PF_NA},
                                 }},
        [PPME_NOTIFICATION_X] = {"NA", EC_UNKNOWN, EF_UNUSED, 0},
        [PPME_SYSCALL_EXECVE_17_E] = {"execve",
                                      EC_PROCESS | EC_SYSCALL,
                                      EF_OLD_VERSION | EF_MODIFIES_STATE | EF_CONVERTER_MANAGED,
                                      0},
        [PPME_SYSCALL_EXECVE_17_X] = {"execve",
                                      EC_PROCESS | EC_SYSCALL,
                                      EF_OLD_VERSION | EF_MODIFIES_STATE | EF_CONVERTER_MANAGED,
                                      17,
                                      {{"res", PT_ERRNO, PF_DEC},
                                       {"exe", PT_CHARBUF, PF_NA},
                                       {"args", PT_BYTEBUF, PF_NA},
                                       {"tid", PT_PID, PF_DEC},
                                       {"pid", PT_PID, PF_DEC},
                                       {"ptid", PT_PID, PF_DEC},
                                       {"cwd", PT_CHARBUF, PF_NA},
                                       {"fdlimit", PT_UINT64, PF_DEC},
                                       {"pgft_maj", PT_UINT64, PF_DEC},
                                       {"pgft_min", PT_UINT64, PF_DEC},
                                       {"vm_size", PT_UINT32, PF_DEC},
                                       {"vm_rss", PT_UINT32, PF_DEC},
                                       {"vm_swap", PT_UINT32, PF_DEC},
                                       {"comm", PT_CHARBUF, PF_NA},
                                       {"cgroups", PT_BYTEBUF, PF_NA},
                                       {"env", PT_BYTEBUF, PF_NA},
                                       {"tty", PT_INT32, PF_DEC}}},
        [PPME_SYSCALL_UNSHARE_E] = {"unshare",
                                    EC_PROCESS | EC_SYSCALL,
                                    EF_OLD_VERSION | EF_CONVERTER_MANAGED,
                                    1,
                                    {{"flags", PT_FLAGS32, PF_HEX, clone_flags}}},
        [PPME_SYSCALL_UNSHARE_X] = {"unshare",
                                    EC_PROCESS | EC_SYSCALL,
                                    EF_CONVERTER_MANAGED,
                                    2,
                                    {{"res", PT_ERRNO, PF_DEC},
                                     {"flags", PT_FLAGS32, PF_HEX, clone_flags}}},
        [PPME_INFRASTRUCTURE_EVENT_E] = {"infra",
                                         EC_INTERNAL | EC_METAEVENT,
                                         EF_SKIPPARSERESET,
                                         4,
                                         {{"source", PT_CHARBUF, PF_DEC},
                                          {"name", PT_CHARBUF, PF_NA},
                                          {"description", PT_CHARBUF, PF_NA},
                                          {"scope", PT_CHARBUF, PF_NA}}},
        [PPME_INFRASTRUCTURE_EVENT_X] = {"NA", EC_UNKNOWN, EF_UNUSED, 0},
        [PPME_SYSCALL_EXECVE_18_E] = {"execve",
                                      EC_PROCESS | EC_SYSCALL,
                                      EF_OLD_VERSION | EF_MODIFIES_STATE | EF_CONVERTER_MANAGED,
                                      1,
                                      {{"filename", PT_FSPATH, PF_NA}}},
        [PPME_SYSCALL_EXECVE_18_X] = {"execve",
                                      EC_PROCESS | EC_SYSCALL,
                                      EF_OLD_VERSION | EF_MODIFIES_STATE | EF_CONVERTER_MANAGED,
                                      17,
                                      {{"res", PT_ERRNO, PF_DEC},
                                       {"exe", PT_CHARBUF, PF_NA},
                                       {"args", PT_BYTEBUF, PF_NA},
                                       {"tid", PT_PID, PF_DEC},
                                       {"pid", PT_PID, PF_DEC},
                                       {"ptid", PT_PID, PF_DEC},
                                       {"cwd", PT_CHARBUF, PF_NA},
                                       {"fdlimit", PT_UINT64, PF_DEC},
                                       {"pgft_maj", PT_UINT64, PF_DEC},
                                       {"pgft_min", PT_UINT64, PF_DEC},
                                       {"vm_size", PT_UINT32, PF_DEC},
                                       {"vm_rss", PT_UINT32, PF_DEC},
                                       {"vm_swap", PT_UINT32, PF_DEC},
                                       {"comm", PT_CHARBUF, PF_NA},
                                       {"cgroups", PT_BYTEBUF, PF_NA},
                                       {"env", PT_BYTEBUF, PF_NA},
                                       {"tty", PT_INT32, PF_DEC}}},
        [PPME_PAGE_FAULT_E] = {"page_fault",
                               EC_OTHER | EC_TRACEPOINT,
                               EF_SKIPPARSERESET,
                               3,
                               {{"addr", PT_UINT64, PF_HEX},
                                {"ip", PT_UINT64, PF_HEX},
                                {"error", PT_FLAGS32, PF_HEX, pf_flags}}},
        [PPME_PAGE_FAULT_X] = {"NA", EC_UNKNOWN, EF_UNUSED, 0},
        [PPME_SYSCALL_EXECVE_19_E] = {"execve",
                                      EC_PROCESS | EC_SYSCALL,
                                      EF_OLD_VERSION | EF_MODIFIES_STATE | EF_CONVERTER_MANAGED,
                                      1,
                                      {{"filename", PT_FSPATH, PF_NA}}},
        [PPME_SYSCALL_EXECVE_19_X] = {"execve",
                                      EC_PROCESS | EC_SYSCALL,
                                      EF_MODIFIES_STATE | EF_CONVERTER_MANAGED,
                                      30,
                                      {{"res", PT_ERRNO, PF_DEC},
                                       {"exe", PT_CHARBUF, PF_NA},
                                       {"args", PT_BYTEBUF, PF_NA},
                                       {"tid", PT_PID, PF_DEC},
                                       {"pid", PT_PID, PF_DEC},
                                       {"ptid", PT_PID, PF_DEC},
                                       {"cwd", PT_CHARBUF, PF_NA},
                                       {"fdlimit", PT_UINT64, PF_DEC},
                                       {"pgft_maj", PT_UINT64, PF_DEC},
                                       {"pgft_min", PT_UINT64, PF_DEC},
                                       {"vm_size", PT_UINT32, PF_DEC},
                                       {"vm_rss", PT_UINT32, PF_DEC},
                                       {"vm_swap", PT_UINT32, PF_DEC},
                                       {"comm", PT_CHARBUF, PF_NA},
                                       {"cgroups", PT_BYTEBUF, PF_NA},
                                       {"env", PT_BYTEBUF, PF_NA},
                                       {"tty", PT_UINT32, PF_DEC},
                                       {"vpgid", PT_PID, PF_DEC},
                                       {"loginuid", PT_UID, PF_DEC},
                                       {"flags", PT_FLAGS32, PF_HEX, execve_flags},
                                       {"cap_inheritable", PT_UINT64, PF_HEX},
                                       {"cap_permitted", PT_UINT64, PF_HEX},
                                       {"cap_effective", PT_UINT64, PF_HEX},
                                       {"exe_ino", PT_UINT64, PF_DEC},
                                       {"exe_ino_ctime", PT_ABSTIME, PF_DEC},
                                       {"exe_ino_mtime", PT_ABSTIME, PF_DEC},
                                       {"uid", PT_UID, PF_DEC},
                                       {"trusted_exepath", PT_FSPATH, PF_NA},
                                       {"pgid", PT_PID, PF_NA},
                                       {"gid", PT_GID, PF_DEC}}},
        [PPME_SYSCALL_SETPGID_E] = {"setpgid",
                                    EC_PROCESS | EC_SYSCALL,
                                    EF_OLD_VERSION | EF_CONVERTER_MANAGED,
                                    2,
                                    {{"pid", PT_PID, PF_DEC}, {"pgid", PT_PID, PF_DEC}}},
        [PPME_SYSCALL_SETPGID_X] = {"setpgid",
                                    EC_PROCESS | EC_SYSCALL,
                                    EF_CONVERTER_MANAGED,
                                    3,
                                    {{"res", PT_ERRNO, PF_DEC},
                                     {"pid", PT_PID, PF_DEC},
                                     {"pgid", PT_PID, PF_DEC}}},
        [PPME_SYSCALL_BPF_E] = {"bpf",
                                EC_OTHER | EC_SYSCALL,
                                EF_OLD_VERSION | EF_CREATES_FD | EF_CONVERTER_MANAGED,
                                1,
                                {{"cmd", PT_INT64, PF_DEC}}},
        [PPME_SYSCALL_BPF_X] =
                {"bpf",
                 EC_OTHER | EC_SYSCALL,
                 EF_OLD_VERSION | EF_CREATES_FD | EF_CONVERTER_MANAGED,
                 1,
                 {{"res_or_fd", PT_DYN, PF_DEC, bpf_dynamic_param, PPM_BPF_IDX_MAX}}},
        [PPME_SYSCALL_SECCOMP_E] = {"seccomp",
                                    EC_OTHER | EC_SYSCALL,
                                    EF_OLD_VERSION | EF_CONVERTER_MANAGED,
                                    2,
                                    {{"op", PT_UINT64, PF_DEC}, {"flags", PT_UINT64, PF_HEX}}},
        [PPME_SYSCALL_SECCOMP_X] = {"seccomp",
                                    EC_OTHER | EC_SYSCALL,
                                    EF_CONVERTER_MANAGED,
                                    3,
                                    {{"res", PT_ERRNO, PF_DEC},
                                     {"op", PT_UINT64, PF_DEC},
                                     {"flags", PT_UINT64, PF_HEX}}},
        [PPME_SYSCALL_UNLINK_2_E] = {"unlink", EC_FILE | EC_SYSCALL, EF_OLD_VERSION, 0},
        [PPME_SYSCALL_UNLINK_2_X] = {"unlink",
                                     EC_FILE | EC_SYSCALL,
                                     EF_NONE,
                                     2,
                                     {{"res", PT_ERRNO, PF_DEC}, {"path", PT_FSPATH, PF_NA}}},
        [PPME_SYSCALL_UNLINKAT_2_E] = {"unlinkat", EC_FILE | EC_SYSCALL, EF_OLD_VERSION, 0},
        [PPME_SYSCALL_UNLINKAT_2_X] = {"unlinkat",
                                       EC_FILE | EC_SYSCALL,
                                       EF_USES_FD,
                                       4,
                                       {{"res", PT_ERRNO, PF_DEC},
                                        {"dirfd", PT_FD, PF_DEC},
                                        {"name", PT_FSRELPATH, PF_NA, DIRFD_PARAM(1)},
                                        {"flags", PT_FLAGS32, PF_HEX, unlinkat_flags}}},
        [PPME_SYSCALL_MKDIRAT_E] = {"mkdirat", EC_FILE | EC_SYSCALL, EF_OLD_VERSION, 0},
        [PPME_SYSCALL_MKDIRAT_X] = {"mkdirat",
                                    EC_FILE | EC_SYSCALL,
                                    EF_USES_FD,
                                    4,
                                    {{"res", PT_ERRNO, PF_DEC},
                                     {"dirfd", PT_FD, PF_DEC},
                                     {"path", PT_FSRELPATH, PF_NA, DIRFD_PARAM(1)},
                                     {"mode", PT_UINT32, PF_HEX}}},
        [PPME_SYSCALL_OPENAT_2_E] = {"openat",
                                     EC_FILE | EC_SYSCALL,
                                     EF_CREATES_FD | EF_MODIFIES_STATE | EF_CONVERTER_MANAGED,
                                     4,
                                     {{"dirfd", PT_FD, PF_DEC},
                                      {"name", PT_FSRELPATH, PF_NA, DIRFD_PARAM(0)},
                                      {"flags", PT_FLAGS32, PF_HEX, file_flags},
                                      {"mode", PT_UINT32, PF_OCT}}},
        [PPME_SYSCALL_OPENAT_2_X] = {"openat",
                                     EC_FILE | EC_SYSCALL,
                                     EF_CREATES_FD | EF_MODIFIES_STATE | EF_CONVERTER_MANAGED,
                                     7,
                                     {{"fd", PT_FD, PF_DEC},
                                      {"dirfd", PT_FD, PF_DEC},
                                      {"name", PT_FSRELPATH, PF_NA, DIRFD_PARAM(1)},
                                      {"flags", PT_FLAGS32, PF_HEX, file_flags},
                                      {"mode", PT_UINT32, PF_OCT},
                                      {"dev", PT_UINT32, PF_HEX},
                                      {"ino", PT_UINT64, PF_DEC}}},
        [PPME_SYSCALL_LINK_2_E] = {"link", EC_FILE | EC_SYSCALL, EF_OLD_VERSION, 0},
        [PPME_SYSCALL_LINK_2_X] = {"link",
                                   EC_FILE | EC_SYSCALL,
                                   EF_NONE,
                                   3,
                                   {{"res", PT_ERRNO, PF_DEC},
                                    {"oldpath", PT_FSPATH, PF_NA},
                                    {"newpath", PT_FSPATH, PF_NA}}},
        [PPME_SYSCALL_LINKAT_2_E] = {"linkat", EC_FILE | EC_SYSCALL, EF_OLD_VERSION, 0},
        [PPME_SYSCALL_LINKAT_2_X] = {"linkat",
                                     EC_FILE | EC_SYSCALL,
                                     EF_NONE,
                                     6,
                                     {{"res", PT_ERRNO, PF_DEC},
                                      {"olddir", PT_FD, PF_DEC},
                                      {"oldpath", PT_FSRELPATH, PF_NA, DIRFD_PARAM(1)},
                                      {"newdir", PT_FD, PF_DEC},
                                      {"newpath", PT_FSRELPATH, PF_NA, DIRFD_PARAM(3)},
                                      {"flags", PT_FLAGS32, PF_HEX, linkat_flags}}},
        [PPME_SYSCALL_FCHMODAT_E] = {"fchmodat", EC_FILE | EC_SYSCALL, EF_OLD_VERSION, 0},
        [PPME_SYSCALL_FCHMODAT_X] = {"fchmodat",
                                     EC_FILE | EC_SYSCALL,
                                     EF_USES_FD,
                                     4,
                                     {{"res", PT_ERRNO, PF_DEC},
                                      {"dirfd", PT_FD, PF_DEC},
                                      {"filename", PT_FSRELPATH, PF_NA, DIRFD_PARAM(1)},
                                      {"mode", PT_MODE, PF_OCT, chmod_mode}}},
        [PPME_SYSCALL_CHMOD_E] = {"chmod", EC_FILE | EC_SYSCALL, EF_OLD_VERSION, 0},
        [PPME_SYSCALL_CHMOD_X] = {"chmod",
                                  EC_FILE | EC_SYSCALL,
                                  EF_NONE,
                                  3,
                                  {{"res", PT_ERRNO, PF_DEC},
                                   {"filename", PT_FSPATH, PF_NA},
                                   {"mode", PT_MODE, PF_OCT, chmod_mode}}},
        [PPME_SYSCALL_FCHMOD_E] = {"fchmod", EC_FILE | EC_SYSCALL, EF_OLD_VERSION, 0},
        [PPME_SYSCALL_FCHMOD_X] = {"fchmod",
                                   EC_FILE | EC_SYSCALL,
                                   EF_USES_FD,
                                   3,
                                   {{"res", PT_ERRNO, PF_DEC},
                                    {"fd", PT_FD, PF_DEC},
                                    {"mode", PT_MODE, PF_OCT, chmod_mode}}},
        [PPME_SYSCALL_RENAMEAT2_E] = {"renameat2", EC_FILE | EC_SYSCALL, EF_OLD_VERSION, 0},
        [PPME_SYSCALL_RENAMEAT2_X] = {"renameat2",
                                      EC_FILE | EC_SYSCALL,
                                      EF_NONE,
                                      6,
                                      {{"res", PT_ERRNO, PF_DEC},
                                       {"olddirfd", PT_FD, PF_DEC},
                                       {"oldpath", PT_FSRELPATH, PF_NA, DIRFD_PARAM(1)},
                                       {"newdirfd", PT_FD, PF_DEC},
                                       {"newpath", PT_FSRELPATH, PF_NA, DIRFD_PARAM(3)},
                                       {"flags", PT_FLAGS32, PF_HEX, renameat2_flags}}},
        [PPME_SYSCALL_USERFAULTFD_E] = {"userfaultfd",
                                        EC_FILE | EC_SYSCALL,
                                        EF_OLD_VERSION | EF_CREATES_FD | EF_MODIFIES_STATE,
                                        0},
        [PPME_SYSCALL_USERFAULTFD_X] = {"userfaultfd",
                                        EC_FILE | EC_SYSCALL,
                                        EF_CREATES_FD | EF_MODIFIES_STATE,
                                        2,
                                        {{"res", PT_ERRNO, PF_DEC},
                                         {"flags", PT_FLAGS32, PF_HEX, file_flags}}},
        [PPME_PLUGINEVENT_E] = {"pluginevent",
                                EC_OTHER | EC_PLUGIN,
                                EF_LARGE_PAYLOAD,
                                2,
                                {{"plugin_id", PT_UINT32, PF_DEC},
                                 {"event_data", PT_BYTEBUF, PF_NA}}},
        [PPME_PLUGINEVENT_X] = {"NA", EC_UNKNOWN, EF_UNUSED, 0},
        [PPME_CONTAINER_JSON_2_E] =
                {"container",
                 EC_PROCESS | EC_METAEVENT,
                 EF_MODIFIES_STATE | EF_LARGE_PAYLOAD,
                 1,
                 {{"json", PT_CHARBUF, PF_NA}}},  /// TODO: do we need SKIPPARSERESET flag?
        [PPME_CONTAINER_JSON_2_X] = {"NA", EC_UNKNOWN, EF_UNUSED, 0},
        [PPME_SYSCALL_OPENAT2_E] = {"openat2",
                                    EC_FILE | EC_SYSCALL,
                                    EF_CREATES_FD | EF_MODIFIES_STATE | EF_CONVERTER_MANAGED,
                                    5,
                                    {{"dirfd", PT_FD, PF_DEC},
                                     {"name", PT_FSRELPATH, PF_NA, DIRFD_PARAM(0)},
                                     {"flags", PT_FLAGS32, PF_HEX, file_flags},
                                     {"mode", PT_UINT32, PF_OCT},
                                     {"resolve", PT_FLAGS32, PF_HEX, openat2_flags}}},
        [PPME_SYSCALL_OPENAT2_X] = {"openat2",
                                    EC_FILE | EC_SYSCALL,
                                    EF_CREATES_FD | EF_MODIFIES_STATE | EF_CONVERTER_MANAGED,
                                    8,
                                    {{"fd", PT_FD, PF_DEC},
                                     {"dirfd", PT_FD, PF_DEC},
                                     {"name", PT_FSRELPATH, PF_NA, DIRFD_PARAM(1)},
                                     {"flags", PT_FLAGS32, PF_HEX, file_flags},
                                     {"mode", PT_UINT32, PF_OCT},
                                     {"resolve", PT_FLAGS32, PF_HEX, openat2_flags},
                                     {"dev", PT_UINT32, PF_HEX},
                                     {"ino", PT_UINT64, PF_DEC}}},
        [PPME_SYSCALL_MPROTECT_E] = {"mprotect",
                                     EC_MEMORY | EC_SYSCALL,
                                     EF_OLD_VERSION | EF_CONVERTER_MANAGED,
                                     3,
                                     {{"addr", PT_UINT64, PF_HEX},
                                      {"length", PT_UINT64, PF_DEC},
                                      {"prot", PT_FLAGS32, PF_HEX, prot_flags}}},
        [PPME_SYSCALL_MPROTECT_X] = {"mprotect",
                                     EC_MEMORY | EC_SYSCALL,
                                     EF_CONVERTER_MANAGED,
                                     4,
                                     {{"res", PT_ERRNO, PF_DEC},
                                      {"addr", PT_UINT64, PF_HEX},
                                      {"length", PT_UINT64, PF_DEC},
                                      {"prot", PT_FLAGS32, PF_HEX, prot_flags}}},
        [PPME_SYSCALL_EXECVEAT_E] = {"execveat",
                                     EC_PROCESS | EC_SYSCALL,
                                     EF_OLD_VERSION | EF_MODIFIES_STATE | EF_CONVERTER_MANAGED,
                                     3,
                                     {{"dirfd", PT_FD, PF_DEC},
                                      {"pathname", PT_FSRELPATH, PF_NA, DIRFD_PARAM(0)},
                                      {"flags", PT_FLAGS32, PF_HEX, execveat_flags}}},
        [PPME_SYSCALL_EXECVEAT_X] = {"execveat",
                                     EC_PROCESS | EC_SYSCALL,
                                     EF_MODIFIES_STATE | EF_CONVERTER_MANAGED,
                                     30,
                                     {{"res", PT_ERRNO, PF_DEC},
                                      {"exe", PT_CHARBUF, PF_NA},
                                      {"args", PT_BYTEBUF, PF_NA},
                                      {"tid", PT_PID, PF_DEC},
                                      {"pid", PT_PID, PF_DEC},
                                      {"ptid", PT_PID, PF_DEC},
                                      {"cwd", PT_CHARBUF, PF_NA},
                                      {"fdlimit", PT_UINT64, PF_DEC},
                                      {"pgft_maj", PT_UINT64, PF_DEC},
                                      {"pgft_min", PT_UINT64, PF_DEC},
                                      {"vm_size", PT_UINT32, PF_DEC},
                                      {"vm_rss", PT_UINT32, PF_DEC},
                                      {"vm_swap", PT_UINT32, PF_DEC},
                                      {"comm", PT_CHARBUF, PF_NA},
                                      {"cgroups", PT_BYTEBUF, PF_NA},
                                      {"env", PT_BYTEBUF, PF_NA},
                                      {"tty", PT_UINT32, PF_DEC},
                                      {"vpgid", PT_PID, PF_DEC},
                                      {"loginuid", PT_UID, PF_DEC},
                                      {"flags", PT_FLAGS32, PF_HEX, execve_flags},
                                      {"cap_inheritable", PT_UINT64, PF_HEX},
                                      {"cap_permitted", PT_UINT64, PF_HEX},
                                      {"cap_effective", PT_UINT64, PF_HEX},
                                      {"exe_ino", PT_UINT64, PF_DEC},
                                      {"exe_ino_ctime", PT_ABSTIME, PF_DEC},
                                      {"exe_ino_mtime", PT_ABSTIME, PF_DEC},
                                      {"uid", PT_UID, PF_DEC},
                                      {"trusted_exepath", PT_FSPATH, PF_NA},
                                      {"pgid", PT_PID, PF_NA},
                                      {"gid", PT_GID, PF_DEC}}},
        [PPME_SYSCALL_COPY_FILE_RANGE_E] = {"copy_file_range",
                                            EC_FILE | EC_SYSCALL,
                                            EF_OLD_VERSION | EF_USES_FD | EF_READS_FROM_FD |
                                                    EF_WRITES_TO_FD | EF_CONVERTER_MANAGED,
                                            3,
                                            {{"fdin", PT_FD, PF_DEC},
                                             {"offin", PT_UINT64, PF_DEC},
                                             {"len", PT_UINT64, PF_DEC}}},
        [PPME_SYSCALL_COPY_FILE_RANGE_X] = {"copy_file_range",
                                            EC_FILE | EC_SYSCALL,
                                            EF_USES_FD | EF_READS_FROM_FD | EF_WRITES_TO_FD |
                                                    EF_CONVERTER_MANAGED,
                                            6,
                                            {{"res", PT_ERRNO, PF_DEC},
                                             {"fdout", PT_FD, PF_DEC},
                                             {"offout", PT_UINT64, PF_DEC},
                                             {"fdin", PT_FD, PF_DEC},
                                             {"offin", PT_UINT64, PF_DEC},
                                             {"len", PT_UINT64, PF_DEC}}},
        [PPME_SYSCALL_CLONE3_E] = {"clone3",
                                   EC_PROCESS | EC_SYSCALL,
                                   EF_OLD_VERSION | EF_MODIFIES_STATE,
                                   0},
        [PPME_SYSCALL_CLONE3_X] = {"clone3",
                                   EC_PROCESS | EC_SYSCALL,
                                   EF_MODIFIES_STATE | EF_CONVERTER_MANAGED,
                                   21,
                                   {{"res", PT_PID, PF_DEC},
                                    {"exe", PT_CHARBUF, PF_NA},
                                    {"args", PT_BYTEBUF, PF_NA},
                                    {"tid", PT_PID, PF_DEC},
                                    {"pid", PT_PID, PF_DEC},
                                    {"ptid", PT_PID, PF_DEC},
                                    {"cwd", PT_CHARBUF, PF_NA},
                                    {"fdlimit", PT_INT64, PF_DEC},
                                    {"pgft_maj", PT_UINT64, PF_DEC},
                                    {"pgft_min", PT_UINT64, PF_DEC},
                                    {"vm_size", PT_UINT32, PF_DEC},
                                    {"vm_rss", PT_UINT32, PF_DEC},
                                    {"vm_swap", PT_UINT32, PF_DEC},
                                    {"comm", PT_CHARBUF, PF_NA},
                                    {"cgroups", PT_BYTEBUF, PF_NA},
                                    {"flags", PT_FLAGS32, PF_HEX, clone_flags},
                                    {"uid", PT_UINT32, PF_DEC},
                                    {"gid", PT_UINT32, PF_DEC},
                                    {"vtid", PT_PID, PF_DEC},
                                    {"vpid", PT_PID, PF_DEC},
                                    {"pidns_init_start_ts", PT_UINT64, PF_DEC}}},
        [PPME_SYSCALL_OPEN_BY_HANDLE_AT_E] = {"open_by_handle_at",
                                              EC_FILE | EC_SYSCALL,
                                              EF_OLD_VERSION | EF_CREATES_FD | EF_MODIFIES_STATE,
                                              0},
        [PPME_SYSCALL_OPEN_BY_HANDLE_AT_X] = {"open_by_handle_at",
                                              EC_FILE | EC_SYSCALL,
                                              EF_CREATES_FD | EF_MODIFIES_STATE,
                                              6,
                                              {{"fd", PT_FD, PF_DEC},
                                               {"mountfd", PT_FD, PF_DEC},
                                               {"flags", PT_FLAGS32, PF_HEX, file_flags},
                                               {"path", PT_FSPATH, PF_NA},
                                               {"dev", PT_UINT32, PF_HEX},
                                               {"ino", PT_UINT64, PF_DEC}}},
        [PPME_SYSCALL_IO_URING_SETUP_E] = {"io_uring_setup",
                                           EC_IO_OTHER | EC_SYSCALL,
                                           EF_OLD_VERSION | EF_CREATES_FD | EF_MODIFIES_STATE,
                                           0},
        [PPME_SYSCALL_IO_URING_SETUP_X] =
                {"io_uring_setup",
                 EC_IO_OTHER | EC_SYSCALL,
                 EF_CREATES_FD | EF_MODIFIES_STATE,
                 8,
                 {{"res", PT_ERRNO, PF_DEC},
                  {"entries", PT_UINT32, PF_DEC},
                  {"sq_entries", PT_UINT32, PF_DEC},
                  {"cq_entries", PT_UINT32, PF_DEC},
                  {"flags", PT_FLAGS32, PF_HEX, io_uring_setup_flags},
                  {"sq_thread_cpu", PT_UINT32, PF_DEC},
                  {"sq_thread_idle", PT_UINT32, PF_DEC},
                  {"features", PT_FLAGS32, PF_HEX, io_uring_setup_feats}}},
        [PPME_SYSCALL_IO_URING_ENTER_E] = {"io_uring_enter",
                                           EC_IO_OTHER | EC_SYSCALL,
                                           EF_OLD_VERSION,
                                           0},
        [PPME_SYSCALL_IO_URING_ENTER_X] = {"io_uring_enter",
                                           EC_IO_OTHER | EC_SYSCALL,
                                           EF_USES_FD,
                                           6,
                                           {{"res", PT_ERRNO, PF_DEC},
                                            {"fd", PT_FD, PF_DEC},
                                            {"to_submit", PT_UINT32, PF_DEC},
                                            {"min_complete", PT_UINT32, PF_DEC},
                                            {"flags", PT_FLAGS32, PF_HEX, io_uring_enter_flags},
                                            {"sig", PT_SIGSET, PF_DEC}}},
        [PPME_SYSCALL_IO_URING_REGISTER_E] = {"io_uring_register",
                                              EC_IO_OTHER | EC_SYSCALL,
                                              EF_OLD_VERSION,
                                              0},
        [PPME_SYSCALL_IO_URING_REGISTER_X] =
                {"io_uring_register",
                 EC_IO_OTHER | EC_SYSCALL,
                 EF_USES_FD,
                 5,
                 {{"res", PT_ERRNO, PF_DEC},
                  {"fd", PT_FD, PF_DEC},
                  {"opcode", PT_ENUMFLAGS16, PF_DEC, io_uring_register_opcodes},
                  {"arg", PT_UINT64, PF_HEX},
                  {"nr_args", PT_UINT32, PF_DEC}}},
        [PPME_SYSCALL_MLOCK_E] = {"mlock", EC_MEMORY | EC_SYSCALL, EF_OLD_VERSION, 0},
        [PPME_SYSCALL_MLOCK_X] = {"mlock",
                                  EC_MEMORY | EC_SYSCALL,
                                  EF_NONE,
                                  3,
                                  {{"res", PT_ERRNO, PF_DEC},
                                   {"addr", PT_UINT64, PF_HEX},
                                   {"len", PT_UINT64, PF_DEC}}},
        [PPME_SYSCALL_MUNLOCK_E] = {"munlock", EC_MEMORY | EC_SYSCALL, EF_OLD_VERSION, 0},
        [PPME_SYSCALL_MUNLOCK_X] = {"munlock",
                                    EC_MEMORY | EC_SYSCALL,
                                    EF_NONE,
                                    3,
                                    {{"res", PT_ERRNO, PF_DEC},
                                     {"addr", PT_UINT64, PF_HEX},
                                     {"len", PT_UINT64, PF_DEC}}},
        [PPME_SYSCALL_MLOCKALL_E] = {"mlockall", EC_MEMORY | EC_SYSCALL, EF_OLD_VERSION, 0},
        [PPME_SYSCALL_MLOCKALL_X] = {"mlockall",
                                     EC_MEMORY | EC_SYSCALL,
                                     EF_NONE,
                                     2,
                                     {{"res", PT_ERRNO, PF_DEC},
                                      {"flags", PT_FLAGS32, PF_HEX, mlockall_flags}}},
        [PPME_SYSCALL_MUNLOCKALL_E] = {"munlockall", EC_MEMORY | EC_SYSCALL, EF_OLD_VERSION, 0},
        [PPME_SYSCALL_MUNLOCKALL_X] =
                {"munlockall", EC_MEMORY | EC_SYSCALL, EF_NONE, 1, {{"res", PT_ERRNO, PF_DEC}}},
        [PPME_SYSCALL_CAPSET_E] = {"capset",
                                   EC_PROCESS | EC_SYSCALL,
                                   EF_OLD_VERSION | EF_MODIFIES_STATE,
                                   0},
        [PPME_SYSCALL_CAPSET_X] = {"capset",
                                   EC_PROCESS | EC_SYSCALL,
                                   EF_MODIFIES_STATE,
                                   4,
                                   {{"res", PT_ERRNO, PF_DEC},
                                    {"cap_inheritable", PT_UINT64, PF_HEX},
                                    {"cap_permitted", PT_UINT64, PF_HEX},
                                    {"cap_effective", PT_UINT64, PF_HEX}}},
        [PPME_USER_ADDED_E] = {"useradded",
                               EC_PROCESS | EC_METAEVENT,
                               EF_MODIFIES_STATE,
                               6,
                               {{"uid", PT_UINT32, PF_DEC},
                                {"gid", PT_UINT32, PF_DEC},
                                {"name", PT_CHARBUF, PF_NA},
                                {"home", PT_CHARBUF, PF_NA},
                                {"shell", PT_CHARBUF, PF_NA},
                                {"container_id", PT_CHARBUF, PF_NA}}},
        [PPME_USER_ADDED_X] = {"NA", EC_UNKNOWN, EF_UNUSED, 0},
        [PPME_USER_DELETED_E] = {"userdeleted",
                                 EC_PROCESS | EC_METAEVENT,
                                 EF_MODIFIES_STATE,
                                 6,
                                 {{"uid", PT_UINT32, PF_DEC},
                                  {"gid", PT_UINT32, PF_DEC},
                                  {"name", PT_CHARBUF, PF_NA},
                                  {"home", PT_CHARBUF, PF_NA},
                                  {"shell", PT_CHARBUF, PF_NA},
                                  {"container_id", PT_CHARBUF, PF_NA}}},
        [PPME_USER_DELETED_X] = {"NA", EC_UNKNOWN, EF_UNUSED, 0},
        [PPME_GROUP_ADDED_E] = {"groupadded",
                                EC_PROCESS | EC_METAEVENT,
                                EF_MODIFIES_STATE,
                                3,
                                {{"gid", PT_UINT32, PF_DEC},
                                 {"name", PT_CHARBUF, PF_NA},
                                 {"container_id", PT_CHARBUF, PF_NA}}},
        [PPME_GROUP_ADDED_X] = {"NA", EC_UNKNOWN, EF_UNUSED, 0},
        [PPME_GROUP_DELETED_E] = {"groupdeleted",
                                  EC_PROCESS | EC_METAEVENT,
                                  EF_MODIFIES_STATE,
                                  3,
                                  {{"gid", PT_UINT32, PF_DEC},
                                   {"name", PT_CHARBUF, PF_NA},
                                   {"container_id", PT_CHARBUF, PF_NA}}},
        [PPME_GROUP_DELETED_X] = {"NA", EC_UNKNOWN, EF_UNUSED, 0},
        [PPME_SYSCALL_DUP2_E] = {"dup2",
                                 EC_IO_OTHER | EC_SYSCALL,
                                 EF_OLD_VERSION | EF_CREATES_FD | EF_USES_FD | EF_MODIFIES_STATE,
                                 1,
                                 {{"fd", PT_FD, PF_DEC}}},
        [PPME_SYSCALL_DUP2_X] = {"dup2",
                                 EC_IO_OTHER | EC_SYSCALL,
                                 EF_CREATES_FD | EF_USES_FD | EF_MODIFIES_STATE,
                                 3,
                                 {{"res", PT_FD, PF_DEC},
                                  {"oldfd", PT_FD, PF_DEC},
                                  {"newfd", PT_FD, PF_DEC}}},
        [PPME_SYSCALL_DUP3_E] = {"dup3",
                                 EC_IO_OTHER | EC_SYSCALL,
                                 EF_OLD_VERSION | EF_CREATES_FD | EF_USES_FD | EF_MODIFIES_STATE,
                                 1,
                                 {{"fd", PT_FD, PF_DEC}}},
        [PPME_SYSCALL_DUP3_X] = {"dup3",
                                 EC_IO_OTHER | EC_SYSCALL,
                                 EF_CREATES_FD | EF_USES_FD | EF_MODIFIES_STATE,
                                 4,
                                 {{"res", PT_FD, PF_DEC},
                                  {"oldfd", PT_FD, PF_DEC},
                                  {"newfd", PT_FD, PF_DEC},
                                  {"flags", PT_FLAGS32, PF_HEX, file_flags}}},
        [PPME_SYSCALL_DUP_1_E] = {"dup",
                                  EC_IO_OTHER | EC_SYSCALL,
                                  EF_OLD_VERSION | EF_CREATES_FD | EF_USES_FD | EF_MODIFIES_STATE,
                                  1,
                                  {{"fd", PT_FD, PF_DEC}}},
        [PPME_SYSCALL_DUP_1_X] = {"dup",
                                  EC_IO_OTHER | EC_SYSCALL,
                                  EF_CREATES_FD | EF_USES_FD | EF_MODIFIES_STATE,
                                  2,
                                  {{"res", PT_FD, PF_DEC}, {"oldfd", PT_FD, PF_DEC}}},
        [PPME_SYSCALL_BPF_2_E] = {"bpf",
                                  EC_OTHER | EC_SYSCALL,
                                  EF_OLD_VERSION | EF_CREATES_FD,
                                  1,
                                  {{"cmd", PT_INT64, PF_DEC}}},
        [PPME_SYSCALL_BPF_2_X] = {"bpf",
                                  EC_OTHER | EC_SYSCALL,
                                  EF_CREATES_FD,
                                  2,
                                  {{"fd", PT_FD, PF_DEC},
                                   {"cmd", PT_ENUMFLAGS32, PF_DEC, bpf_commands}}},
        [PPME_SYSCALL_MLOCK2_E] = {"mlock2", EC_MEMORY | EC_SYSCALL, EF_OLD_VERSION, 0},
        [PPME_SYSCALL_MLOCK2_X] = {"mlock2",
                                   EC_MEMORY | EC_SYSCALL,
                                   EF_NONE,
                                   4,
                                   {{"res", PT_ERRNO, PF_DEC},
                                    {"addr", PT_UINT64, PF_HEX},
                                    {"len", PT_UINT64, PF_DEC},
                                    {"flags", PT_FLAGS32, PF_HEX, mlock2_flags}}},
        [PPME_SYSCALL_FSCONFIG_E] = {"fsconfig", EC_SYSTEM | EC_SYSCALL, EF_OLD_VERSION, 0},
        [PPME_SYSCALL_FSCONFIG_X] = {"fsconfig",
                                     EC_SYSTEM | EC_SYSCALL,
                                     EF_USES_FD,
                                     7,
                                     {{"res", PT_ERRNO, PF_DEC},
                                      {"fd", PT_FD, PF_DEC},
                                      {"cmd", PT_ENUMFLAGS32, PF_DEC, fsconfig_cmds},
                                      {"key", PT_CHARBUF, PF_NA},
                                      {"value_bytebuf", PT_BYTEBUF, PF_NA},
                                      {"value_charbuf", PT_CHARBUF, PF_NA},
                                      {"aux", PT_INT32, PF_DEC}}},
        [PPME_SYSCALL_EPOLL_CREATE_E] = {"epoll_create",
                                         EC_WAIT | EC_SYSCALL,
                                         EF_OLD_VERSION | EF_CREATES_FD | EF_MODIFIES_STATE |
                                                 EF_CONVERTER_MANAGED,
                                         1,
                                         {{"size", PT_INT32, PF_DEC}}},
        [PPME_SYSCALL_EPOLL_CREATE_X] = {"epoll_create",
                                         EC_WAIT | EC_SYSCALL,
                                         EF_CREATES_FD | EF_MODIFIES_STATE | EF_CONVERTER_MANAGED,
                                         2,
                                         {{"res", PT_ERRNO, PF_DEC}, {"size", PT_INT32, PF_DEC}}},
        [PPME_SYSCALL_EPOLL_CREATE1_E] = {"epoll_create1",
                                          EC_WAIT | EC_SYSCALL,
                                          EF_OLD_VERSION | EF_CREATES_FD | EF_MODIFIES_STATE |
                                                  EF_CONVERTER_MANAGED,
                                          1,
                                          {{"flags", PT_FLAGS32, PF_HEX, epoll_create1_flags}}},
        [PPME_SYSCALL_EPOLL_CREATE1_X] = {"epoll_create1",
                                          EC_WAIT | EC_SYSCALL,
                                          EF_CREATES_FD | EF_MODIFIES_STATE | EF_CONVERTER_MANAGED,
                                          2,
                                          {{"res", PT_ERRNO, PF_DEC},
                                           {"flags", PT_FLAGS32, PF_HEX, epoll_create1_flags}}},
        [PPME_SYSCALL_CHOWN_E] = {"chown", EC_FILE | EC_SYSCALL, EF_OLD_VERSION, 0},
        [PPME_SYSCALL_CHOWN_X] = {"chown",
                                  EC_FILE | EC_SYSCALL,
                                  EF_NONE,
                                  4,
                                  {{"res", PT_ERRNO, PF_DEC},
                                   {"path", PT_FSPATH, PF_NA},
                                   {"uid", PT_UINT32, PF_DEC},
                                   {"gid", PT_UINT32, PF_DEC}}},
        [PPME_SYSCALL_LCHOWN_E] = {"lchown", EC_FILE | EC_SYSCALL, EF_OLD_VERSION, 0},
        [PPME_SYSCALL_LCHOWN_X] = {"lchown",
                                   EC_FILE | EC_SYSCALL,
                                   EF_NONE,
                                   4,
                                   {{"res", PT_ERRNO, PF_DEC},
                                    {"path", PT_FSPATH, PF_NA},
                                    {"uid", PT_UINT32, PF_DEC},
                                    {"gid", PT_UINT32, PF_DEC}}},
        [PPME_SYSCALL_FCHOWN_E] = {"fchown", EC_FILE | EC_SYSCALL, EF_OLD_VERSION, 0},
        [PPME_SYSCALL_FCHOWN_X] = {"fchown",
                                   EC_FILE | EC_SYSCALL,
                                   EF_USES_FD,
                                   4,
                                   {{"res", PT_ERRNO, PF_DEC},
                                    {"fd", PT_FD, PF_DEC},
                                    {"uid", PT_UINT32, PF_DEC},
                                    {"gid", PT_UINT32, PF_DEC}}},
        [PPME_SYSCALL_FCHOWNAT_E] = {"fchownat", EC_FILE | EC_SYSCALL, EF_OLD_VERSION, 0},
        [PPME_SYSCALL_FCHOWNAT_X] = {"fchownat",
                                     EC_FILE | EC_SYSCALL,
                                     EF_USES_FD,
                                     6,
                                     {{"res", PT_ERRNO, PF_DEC},
                                      {"dirfd", PT_FD, PF_DEC},
                                      {"pathname", PT_FSRELPATH, PF_NA, DIRFD_PARAM(1)},
                                      {"uid", PT_UINT32, PF_DEC},
                                      {"gid", PT_UINT32, PF_DEC},
                                      {"flags", PT_FLAGS32, PF_HEX, fchownat_flags}}},
        [PPME_SYSCALL_UMOUNT_1_E] = {"umount",
                                     EC_FILE | EC_SYSCALL,
                                     EF_OLD_VERSION | EF_MODIFIES_STATE,
                                     0},
        [PPME_SYSCALL_UMOUNT_1_X] = {"umount",
                                     EC_FILE | EC_SYSCALL,
                                     EF_MODIFIES_STATE,
                                     2,
                                     {{"res", PT_ERRNO, PF_DEC}, {"name", PT_FSPATH, PF_NA}}},
        [PPME_SOCKET_ACCEPT4_6_E] = {"accept4",
                                     EC_NET | EC_SYSCALL,
                                     EF_OLD_VERSION | EF_CREATES_FD | EF_MODIFIES_STATE |
                                             EF_CONVERTER_MANAGED,
                                     1,
                                     {{"flags", PT_INT32, PF_HEX}}},
        [PPME_SOCKET_ACCEPT4_6_X] = {"accept4",
                                     EC_NET | EC_SYSCALL,
                                     EF_CREATES_FD | EF_MODIFIES_STATE | EF_CONVERTER_MANAGED,
                                     6,
                                     {{"fd", PT_FD, PF_DEC},
                                      {"tuple", PT_SOCKTUPLE, PF_NA},
                                      {"queuepct", PT_UINT8, PF_DEC},
                                      {"queuelen", PT_UINT32, PF_DEC},
                                      {"queuemax", PT_UINT32, PF_DEC},
                                      {"flags", PT_INT32, PF_HEX}}},
        [PPME_SYSCALL_UMOUNT2_E] = {"umount2",
                                    EC_FILE | EC_SYSCALL,
                                    EF_OLD_VERSION | EF_MODIFIES_STATE | EF_CONVERTER_MANAGED,
                                    1,
                                    {{"flags", PT_FLAGS32, PF_HEX, umount_flags}}},
        [PPME_SYSCALL_UMOUNT2_X] = {"umount2",
                                    EC_FILE | EC_SYSCALL,
                                    EF_MODIFIES_STATE | EF_CONVERTER_MANAGED,
                                    3,
                                    {{"res", PT_ERRNO, PF_DEC},
                                     {"name", PT_FSPATH, PF_NA},
                                     {"flags", PT_FLAGS32, PF_HEX, umount_flags}}},
        [PPME_SYSCALL_PIPE2_E] = {"pipe2",
                                  EC_IPC | EC_SYSCALL,
                                  EF_OLD_VERSION | EF_CREATES_FD | EF_MODIFIES_STATE,
                                  0},
        [PPME_SYSCALL_PIPE2_X] = {"pipe2",
                                  EC_IPC | EC_SYSCALL,
                                  EF_CREATES_FD | EF_MODIFIES_STATE,
                                  5,
                                  {{"res", PT_ERRNO, PF_DEC},
                                   {"fd1", PT_FD, PF_DEC},
                                   {"fd2", PT_FD, PF_DEC},
                                   {"ino", PT_UINT64, PF_DEC},
                                   {"flags", PT_FLAGS32, PF_HEX, file_flags}}},
        [PPME_SYSCALL_INOTIFY_INIT1_E] = {"inotify_init1",
                                          EC_IPC | EC_SYSCALL,
                                          EF_OLD_VERSION | EF_CREATES_FD | EF_MODIFIES_STATE,
                                          0},
        [PPME_SYSCALL_INOTIFY_INIT1_X] = {"inotify_init1",
                                          EC_IPC | EC_SYSCALL,
                                          EF_CREATES_FD | EF_MODIFIES_STATE,
                                          2,
                                          {{"res", PT_FD, PF_DEC},
                                           {"flags", PT_FLAGS16, PF_HEX, file_flags}}},
        [PPME_SYSCALL_EVENTFD2_E] = {"eventfd2",
                                     EC_IPC | EC_SYSCALL,
                                     EF_OLD_VERSION | EF_CREATES_FD | EF_MODIFIES_STATE |
                                             EF_CONVERTER_MANAGED,
                                     1,
                                     {{"initval", PT_UINT64, PF_DEC}}},
        [PPME_SYSCALL_EVENTFD2_X] = {"eventfd2",
                                     EC_IPC | EC_SYSCALL,
                                     EF_CREATES_FD | EF_MODIFIES_STATE | EF_CONVERTER_MANAGED,
                                     3,
                                     {{"res", PT_FD, PF_DEC},
                                      {"flags", PT_FLAGS16, PF_HEX, file_flags},
                                      {"initval", PT_UINT64, PF_DEC}}},
        [PPME_SYSCALL_SIGNALFD4_E] = {"signalfd4",
                                      EC_SIGNAL | EC_SYSCALL,
                                      EF_OLD_VERSION | EF_CREATES_FD | EF_MODIFIES_STATE |
                                              EF_CONVERTER_MANAGED,
                                      2,
                                      {{"fd", PT_FD, PF_DEC}, {"mask", PT_UINT32, PF_HEX}}},
        [PPME_SYSCALL_SIGNALFD4_X] = {"signalfd4",
                                      EC_SIGNAL | EC_SYSCALL,
                                      EF_CREATES_FD | EF_MODIFIES_STATE | EF_CONVERTER_MANAGED,
                                      4,
                                      {{"res", PT_FD, PF_DEC},
                                       {"flags", PT_FLAGS16, PF_HEX, file_flags},
                                       {"fd", PT_FD, PF_DEC},
                                       {"mask", PT_UINT32, PF_HEX}}},
        [PPME_SYSCALL_PRCTL_E] = {"prctl",
                                  EC_PROCESS | EC_SYSCALL,
                                  EF_OLD_VERSION | EF_MODIFIES_STATE,
                                  0},
        [PPME_SYSCALL_PRCTL_X] = {"prctl",
                                  EC_PROCESS | EC_SYSCALL,
                                  EF_MODIFIES_STATE,
                                  4,
                                  {{"res", PT_ERRNO, PF_DEC},
                                   {"option", PT_ENUMFLAGS32, PF_DEC, prctl_options},
                                   {"arg2_str", PT_CHARBUF, PF_NA},
                                   {"arg2_int", PT_INT64, PF_DEC}}},
        [PPME_ASYNCEVENT_E] = {"asyncevent",
                               EC_OTHER | EC_METAEVENT,
                               EF_LARGE_PAYLOAD,
                               3,
                               {{"plugin_id", PT_UINT32, PF_DEC},
                                {"name", PT_CHARBUF, PF_NA},
                                {"data", PT_BYTEBUF, PF_NA}}},
        [PPME_ASYNCEVENT_X] = {"NA", EC_UNKNOWN, EF_UNUSED, 0},
        [PPME_SYSCALL_MEMFD_CREATE_E] = {"memfd_create",
                                         EC_MEMORY | EC_SYSCALL,
                                         EF_OLD_VERSION | EF_CREATES_FD | EF_MODIFIES_STATE,
                                         0},
        [PPME_SYSCALL_MEMFD_CREATE_X] = {"memfd_create",
                                         EC_MEMORY | EC_SYSCALL,
                                         EF_CREATES_FD | EF_MODIFIES_STATE,
                                         3,
                                         {{"fd", PT_FD, PF_DEC},
                                          {"name", PT_CHARBUF, PF_NA},
                                          {"flags", PT_FLAGS32, PF_HEX, memfd_create_flags}}},
        [PPME_SYSCALL_PIDFD_GETFD_E] = {"pidfd_getfd",
                                        EC_PROCESS | EC_SYSCALL,
                                        EF_OLD_VERSION | EF_CREATES_FD | EF_MODIFIES_STATE,
                                        0},
        [PPME_SYSCALL_PIDFD_GETFD_X] = {"pidfd_getfd",
                                        EC_PROCESS | EC_SYSCALL,
                                        EF_CREATES_FD | EF_MODIFIES_STATE,
                                        4,
                                        {{"fd", PT_FD, PF_DEC},
                                         {"pid_fd", PT_FD, PF_DEC},
                                         {"target_fd", PT_FD, PF_DEC},
                                         {"flags", PT_UINT32, PF_HEX}}},
        [PPME_SYSCALL_PIDFD_OPEN_E] = {"pidfd_open",
                                       EC_PROCESS | EC_SYSCALL,
                                       EF_OLD_VERSION | EF_CREATES_FD | EF_MODIFIES_STATE,
                                       0},
        [PPME_SYSCALL_PIDFD_OPEN_X] = {"pidfd_open",
                                       EC_PROCESS | EC_SYSCALL,
                                       EF_CREATES_FD | EF_MODIFIES_STATE,
                                       3,
                                       {{"fd", PT_FD, PF_DEC},
                                        {"pid", PT_PID, PF_DEC},
                                        {"flags", PT_FLAGS32, PF_HEX, pidfd_open_flags}}},
        [PPME_SYSCALL_INIT_MODULE_E] = {"init_module", EC_OTHER | EC_SYSCALL, EF_OLD_VERSION, 0},
        [PPME_SYSCALL_INIT_MODULE_X] = {"init_module",
                                        EC_OTHER | EC_SYSCALL,
                                        EF_NONE,
                                        4,
                                        {{"res", PT_ERRNO, PF_DEC},
                                         {"img", PT_BYTEBUF, PF_NA},
                                         {"length", PT_UINT64, PF_DEC},
                                         {"uargs", PT_CHARBUF, PF_NA}}},
        [PPME_SYSCALL_FINIT_MODULE_E] = {"finit_module", EC_OTHER | EC_SYSCALL, EF_OLD_VERSION, 0},
        [PPME_SYSCALL_FINIT_MODULE_X] = {"finit_module",
                                         EC_OTHER | EC_SYSCALL,
                                         EF_USES_FD | EF_READS_FROM_FD,
                                         4,
                                         {{"res", PT_ERRNO, PF_DEC},
                                          {"fd", PT_FD, PF_DEC},
                                          {"uargs", PT_CHARBUF, PF_NA},
                                          {"flags", PT_FLAGS32, PF_HEX, finit_module_flags}}},
        [PPME_SYSCALL_MKNOD_E] = {"mknod", EC_OTHER | EC_SYSCALL, EF_OLD_VERSION, 0},
        [PPME_SYSCALL_MKNOD_X] = {"mknod",
                                  EC_OTHER | EC_SYSCALL,
                                  EF_NONE,
                                  4,
                                  {{"res", PT_ERRNO, PF_DEC},
                                   {"path", PT_FSPATH, PF_NA},
                                   {"mode", PT_MODE, PF_OCT, mknod_mode},
                                   {"dev", PT_UINT32, PF_DEC}}},
        [PPME_SYSCALL_MKNODAT_E] = {"mknodat", EC_OTHER | EC_SYSCALL, EF_OLD_VERSION, 0},
        [PPME_SYSCALL_MKNODAT_X] = {"mknodat",
                                    EC_OTHER | EC_SYSCALL,
                                    EF_USES_FD,
                                    5,
                                    {{"res", PT_ERRNO, PF_DEC},
                                     {"dirfd", PT_FD, PF_DEC},
                                     {"path", PT_FSRELPATH, PF_NA, DIRFD_PARAM(1)},
                                     {"mode", PT_MODE, PF_OCT, mknod_mode},
                                     {"dev", PT_UINT32, PF_DEC}}},
        [PPME_SYSCALL_NEWFSTATAT_E] = {"newfstatat", EC_FILE | EC_SYSCALL, EF_OLD_VERSION, 0},
        [PPME_SYSCALL_NEWFSTATAT_X] = {"newfstatat",
                                       EC_FILE | EC_SYSCALL,
                                       EF_USES_FD,
                                       4,
                                       {{"res", PT_ERRNO, PF_DEC},
                                        {"dirfd", PT_FD, PF_DEC},
                                        {"path", PT_FSRELPATH, PF_NA, DIRFD_PARAM(1)},
                                        {"flags", PT_FLAGS32, PF_HEX, newfstatat_flags}}},
        [PPME_SYSCALL_PROCESS_VM_READV_E] = {"process_vm_readv",
                                             EC_SYSCALL | EC_IPC,
                                             EF_OLD_VERSION,
                                             0},
        [PPME_SYSCALL_PROCESS_VM_READV_X] = {"process_vm_readv",
                                             EC_SYSCALL | EC_IPC,
                                             EF_NONE,
                                             3,
                                             {{"res", PT_INT64, PF_DEC},
                                              {"pid", PT_PID, PF_DEC},
                                              {"data", PT_BYTEBUF, PF_NA}}},
        [PPME_SYSCALL_PROCESS_VM_WRITEV_E] = {"process_vm_writev",
                                              EC_SYSCALL | EC_IPC,
                                              EF_OLD_VERSION,
                                              0},
        [PPME_SYSCALL_PROCESS_VM_WRITEV_X] = {"process_vm_writev",
                                              EC_SYSCALL | EC_IPC,
                                              EF_NONE,
                                              3,
                                              {{"res", PT_INT64, PF_DEC},
                                               {"pid", PT_PID, PF_DEC},
                                               {"data", PT_BYTEBUF, PF_NA}}},
        [PPME_SYSCALL_DELETE_MODULE_E] = {"delete_module",
                                          EC_OTHER | EC_SYSCALL,
                                          EF_OLD_VERSION,
                                          0},
        [PPME_SYSCALL_DELETE_MODULE_X] = {"delete_module",
                                          EC_OTHER | EC_SYSCALL,
                                          EF_NONE,
                                          3,
                                          {{"res", PT_ERRNO, PF_DEC},
                                           {"name", PT_CHARBUF, PF_NA},
                                           {"flags", PT_FLAGS32, PF_HEX, delete_module_flags}}},
        [PPME_SYSCALL_SETREUID_E] = {"setreuid",
                                     EC_USER | EC_SYSCALL,
                                     EF_OLD_VERSION | EF_MODIFIES_STATE,
                                     0},
        [PPME_SYSCALL_SETREUID_X] = {"setreuid",
                                     EC_USER | EC_SYSCALL,
                                     EF_MODIFIES_STATE,
                                     3,
                                     {{"res", PT_ERRNO, PF_DEC},
                                      {"ruid", PT_UID, PF_DEC},
                                      {"euid", PT_UID, PF_DEC}}},
        [PPME_SYSCALL_SETREGID_E] = {"setregid", EC_USER | EC_SYSCALL, EF_OLD_VERSION, 0},
        [PPME_SYSCALL_SETREGID_X] = {"setregid",
                                     EC_USER | EC_SYSCALL,
                                     EF_MODIFIES_STATE,
                                     3,
                                     {{"res", PT_ERRNO, PF_DEC},
                                      {"rgid", PT_UID, PF_DEC},
                                      {"egid", PT_UID, PF_DEC}}},
};
#pragma GCC diagnostic pop

// We don't need this check in kmod (this source file is included during kmod compilation!)
// This also avoids weird situation where the _Static_assert is not available in some very old
// compilers, thus breaking the kmod build.
#ifndef __KERNEL__
// This code is compiled on windows and osx too!
// Make sure to be on gcc or that the c standard is >= c11
#if defined __GNUC__ || __STDC_VERSION__ >= 201112L
_Static_assert(sizeof(g_event_info) / sizeof(*g_event_info) == PPM_EVENT_MAX,
               "Missing event entries in event table.");
#endif
#endif
