/***************************************************************************
 *                                                                         *
 *                         Powersave Daemon                                *
 *                                                                         *
 *          Copyright (C) 2004,2005 SUSE Linux Products GmbH               *
 *                                                                         *
 * This program is free software; you can redistribute it and/or modify it *
 * under the terms of the GNU General Public License as published by the   *
 * Free Software Foundation; either version 2 of the License, or (at you   *
 * option) any later version.                                              *
 *                                                                         *
 * This program is distributed in the hope that it will be useful, but     *
 * WITHOUT ANY WARRANTY; without even the implied warranty of              *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU       *
 * General Public License for more details.                                *
 *                                                                         *
 * You should have received a copy of the GNU General Public License along *
 * with this program; if not, write to the Free Software Foundation, Inc., *
 * 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA                  *
 *                                                                         *
 ***************************************************************************/

#include "config.h"
#include <unistd.h>
#include <sys/types.h>
#include <dirent.h>
#include <stdio.h>
#include <errno.h>
#include <string.h>
#include <stdarg.h>
#include <ctype.h>

#include "powerlib_local.h"
#include "powerlib.h"

/* toDo: make this function inline somehow ...*/
/*inline*/
#define ACPI_PROC "/proc/acpi"
#define APM_PROC "/proc/apm"

static int isACPI = NOT_SUPPORTED;

int checkACPI()
{
	/* already set, so don't check it again */
	if (isACPI != NOT_SUPPORTED)
		return isACPI;
	if (!access(ACPI_PROC, R_OK)) {
		isACPI = ACPI;
		return isACPI;
	}
	if (access(APM_PROC, R_OK)) {
		isACPI = NOT_SUPPORTED;
		return isACPI;
	} else {
		isACPI = APM;
		return isACPI;
	}
}

int getDevicesNum(const char *dir_name)
{
	DIR *dir;
	struct dirent *dirent;
	int no = 0;

	dir = opendir(dir_name);
	if (!dir) {
		pDebug(DBG_DIAG, "opendir(%s): %s", dir_name, strerror(errno));
		return -1;
	}

	/* scan all the battery directories (e.g. BAT0, ...) */
	/* expect that dirent will always give back the same order of files(directories) */
	while ((dirent = readdir(dir)) != NULL) {
		if (dirent->d_name[0] == '.')
			continue;	/* skip dotfiles */
		no++;
		pDebug(DBG_INFO, "Device %s found in system.", dirent->d_name);
	}
	closedir(dir);
	return no;
}

int getColonValue(FILE * file, char *value, const size_t value_size, char *def, const size_t def_size)
{
	char line[MAX_LINE_SIZE + 1] = "";
	char *no;
	int slen, y, x, k;

	if (file == NULL) {
		goto ERROR;
	}
	
	no = fgets(line, sizeof(line), file);
	if (!no) {
		goto ERROR;
	}
	slen = strlen(line);
	/* We parse two strings here divided by a colon
	   Each string may have multiple words (spaces in between)
	   Spaces and tabs before and after each string are stripped */
	for (x = 0; x < MAX_LINE_SIZE && line[x] != ':' && line[x] != '\0' && line[x] != '\n'; x++) ;
	if (x >= MAX_LINE_SIZE || line[x] != ':') {
		/* no colon */
		goto ERROR;
	}
	if (x >= def_size) {
		goto ERROR;
	}
	/* empty line */
	if (x < 2 || isspace(line[0]))
		return -1;
	/*strip away all spaces after the left string */
	for (y = x - 1; y >= 0 && isspace(line[y]); y--) ;
	y++;
	memcpy(def, line, y);
	*(def + y) = '\0';
	/*      pDebug(DBG_DIAG, "First String: -%s- k: %d; y: %d;x %d", def, k, y, x); */
	x++;
	/*strip away all spaces before the right string */
	for (k = x; k < slen && k < MAX_LINE_SIZE && isspace(line[k]); k++) ;
	for (x = k; x < slen && x < MAX_LINE_SIZE && *(line + x) != '\n' && *(line + x) != '\0'; x++) ;
	/*strip away spaces after right string and before end of line */
	for (y = slen - 1; y >= x && isspace(line[y]); y--) ;
	y++;
	if (y - k >= value_size) {
		pDebug(DBG_DIAG, "String too long (%d): '%s'", (int)value_size, line);
		goto ERROR;
	}
	memcpy(value, line + k, y - k);
	*(value + (y - k)) = '\0';
	return 1;
	
ERROR:
	if ( value_size > strlen("unknown"))
		strcpy(value,"unknown");
	return -1;
}

int getDirEntry(const int num, char *file, const size_t size_file, const char *dir_in)
{

	DIR *dir;
	struct dirent *dirent;

	int dir_count = 0;

	dir = opendir(dir_in);
	if (!dir) {
		pDebug(DBG_DIAG, "Could not open directory %s: %s", dir_in, strerror(errno));
		return -1;
	}

	/* get the num file of directory dir */
	/* expect that dirent will always give back the same order of files(directories) */
	while ((dirent = readdir(dir))) {
		if (dirent->d_name[0] == '.')
			continue;	/* skip dotfiles */
		if (dir_count == num)
			break;
		dir_count++;
	}
	if (!file || !dirent || !dirent->d_name) {
		/* Do not write an error here, device does not exist, DSDT error,
		   or device not supported to be handled through ACPI, maybe there
		   is simply e.g. no battery at all...

		   pDebug(DBG_DIAG, "%d. entry could not be returned"
		   " for directory: %s\n", dir_count, dir_in); */

		closedir(dir);
		return -1;
	}
	if (strlen(dirent->d_name) > size_file) {
		pDebug(DBG_DIAG, "%d. could not be returned for directory: %s"
		       " because file (size: %Zd) is not large enough", dir_count, dir_in, size_file);
	}
	strcpy(file, dirent->d_name);
	closedir(dir);
	return 1;
}

int _write_line(const char *filename, const char *fmt, ...)
{
	FILE *fp = fopen(filename, "w+");
	if (!fp) {
		pDebug(DBG_DIAG, "Could not open %s for writing", filename);
		return -1;
	}
	va_list ap;
	va_start(ap, fmt);	/* get variable argument list passed */
	if (vfprintf(fp, fmt, ap) < 0) {
		pDebug(DBG_DIAG, "Could not write to %s", filename);
		fclose(fp);
		return -1;
	}
	va_end(ap);
	fclose(fp);
	return 1;
}

inline int check_ACPI_dir(const char *directory)
{

	DIR *dir = NULL;
	struct dirent *dirent = NULL;
	int x;

	dir = opendir(directory);
	if (!dir) {
		pDebug(DBG_DIAG, "Cannot open directory %s: %s", directory, strerror(errno));
		return NO_MODULE_ERROR;
	}

	/* are there battery devices available ? */
	for (x = 0; x < 3; x++) {
		dirent = readdir(dir);
		if (dirent == NULL) {
			closedir(dir);
			return NO_DEVICE_ERROR;
		}
		if (dirent->d_name[0] == '.')
			continue;	/* skip dotfiles */
	}
	closedir(dir);
	if (x == 3)
		return 1;
	else
		return NO_DEVICE_ERROR;
}


char *DBus_Error_Array[REPLY_ERROR_MAX] = {
	"success",
	"no connection",
	"no rights",
	"invalid param",
	"invalid method",
	"invalid interface",
	"not supported",
	"disabled",
	"already set",
	"dbus error",
	"general error",
	"dbus invalid message type",
};
