/***************************************************************************
 *                                                                         *
 *                         Powersave Daemon                                *
 *                                                                         *
 *          Copyright (C) 2004,2005 SUSE Linux Products GmbH               *
 *                                                                         *
 *               Author(s): Holger Macht <hmacht@suse.de>                  *
 *                                                                         *
 * This program is free software; you can redistribute it and/or modify it *
 * under the terms of the GNU General Public License as published by the   *
 * Free Software Foundation; either version 2 of the License, or (at you   *
 * option) any later version.                                              *
 *                                                                         *
 * This program is distributed in the hope that it will be useful, but     *
 * WITHOUT ANY WARRANTY; without even the implied warranty of              *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU       *
 * General Public License for more details.                                *
 *                                                                         *
 * You should have received a copy of the GNU General Public License along *
 * with this program; if not, write to the Free Software Foundation, Inc., *
 * 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA                  *
 *                                                                         *
 ***************************************************************************/

#include "config.h"
#include <getopt.h>
#include <sys/wait.h>
#include <sys/fcntl.h>
#include <fstream>

#include "config_pm.h"
#include "acpi.h"
#include "apm.h"
#include "main_loop.h"
#include "cpufreq_management.h"
#include "globals.h"

namespace Powersave {
	namespace Globals {
		GeneralConfig *config_obj;
		PM_Interface *pm;
		MainLoop *main_loop;
		CpufreqManagement *cpufreq;
	};
};

using namespace Powersave::Globals;

#define POWERSAVED_PID_FILE_PATH "/var/run/powersaved.pid"
/** @brief the maximum length of a param including paths */
#define MAX_PARAM_LENGTH 128


// I currently destroy the object I am still working on.
// Don't know yet if this is proper but should free all resources
void exit_handler(int)
{
	// unlink files here, I do not know if there is already a pm object!
	unlink(POWERSAVED_PID_FILE_PATH);
	if (!pm) {
		// this is not an error!
		pDebug(DBG_WARN, "Main object not created yet,"
		       "program seems to be terminated shortly after starting");
		exit(0);
	} else {
		delete pm;
		sleep(1);
		exit(0);
	}
}

int daemonize()
{
	int fdlimit;
	switch (fork()) {
	case 0:
		break;
	case -1:
		return -1;
	default:
		_exit(0);	/* exit the original process */
	}

	if (setsid() < 0)	/* shoudn't fail */
		return -1;

	switch (fork()) {
	case 0:
		break;
	case -1:
		return -1;
	default:
		_exit(0);
	}
 
 	if (chdir("/") < 0) {
 		pDebug(DBG_WARN, "Can not change working directory");
 	}

	// close all fds
	fdlimit = sysconf(_SC_OPEN_MAX);

	for (int fd = 0; fd < fdlimit; fd++)
		close(fd);
	open("/dev/null", O_RDWR);
 
 	if (dup(0) < 0 || dup(0) < 0) {
 		pDebug(DBG_WARN, "Can not duplicate filedescriptor 0");
 	}
	return 0;
}

void USR1_handler(int)
{
	if (pm) {
		pm->resume();
	}
}

void hup_handler(int)
{
	if (!pm) {
		pDebug(DBG_ERR, "Could not re-read configs, there is no main object, this should not happen.");
	} else {
		pDebug(DBG_INFO, "HUP signal occured, going to re-read configs");
		if (pm->rereadConfig() < 0) {
			pDebug(DBG_WARN, "could not re-read all config in HUP signal handler");
		}
		// check whether force battery polling could be disabled in this function
		pm->checkBatteryStateChanges();
	}
}

void usage()
{
	fprintf(stderr, "Usage: powersaved [OPTIONS] \n\
       where OPTIONS can be :\n\
            -f --acpi-file FILE\n\
                 sets the acpi file the daemon can receive ACPI events from.\n\
                 This could either be /proc/acpi/events or the acpi daemon's socket file\n\n\
            -d --daemonize\n\
                 run as a daemon\n\n\
            -x --scripts-dir\n\
                 sets scripts directory where executables that are triggered on \n\
                 events are found (default: %s) \n\n\
            -y --internal-scripts-dir\n\
                 sets scripts directory where executables that are triggered on \n\
                 internal events are found (default: %s) \n\n\
            -c --config-dir dir\n\
                 sets the config directory (default is %s)\n\n\
            -s --scheme-dir dir\n\
                 sets the directory for scheme config files (default is %s)\n\n\
            -h --help\n\
                 print this help\n\n\
            -v --verbosity value\n\
                 valid values are from 0-15. \n\
                 1-Error, 2-Warning, 4-Diag, 8-Info and combinations \n\n",
                 PUB_SCRIPT_DIR, SCRIPT_DIR, CONFIG_DIR, CONFIG_DIR);
}

void get_args(int argc, char **argv, GeneralConfig * config_obj, int &daemon, int acpi_apm)
{
	int option_index = 0;
	int verbose = 0;
	char *endptr[MAX_PARAM_LENGTH + 1];
	daemon = 0;

	struct option opts[] = {
		{"daemonize", 0, 0, 'd'},
		{"acpi-file", 0, 0, 'f'},
		{"scheme-dir", 0, 0, 's'},
		{"scripts-dir", 0, 0, 'x'},
		{"internal-scripts-dir", 0, 0, 'y'},
		{"verbosity", 0, 0, 'v'},
		{"config-dir", 0, 0, 'c'},
		{"help", 0, 0, 'h'},
		{NULL, 0, 0, 0},
	};
	while (1) {
		int i = getopt_long(argc, argv, "v:c:s:x:y:f:dh", opts, &option_index);
		if (i == -1) {
			break;
		}
		if (optarg && strlen(optarg) >= MAX_PARAM_LENGTH) {
			fprintf(stderr, "Parameter: '%s' is too long\n", optarg);
			goto exit_failure;
		}
		switch (i) {
		case 0:
			break;
		case 'n':
			config_obj->disable_CPU_freq = 1;
			break;
		case 's':
			if (!access(optarg, R_OK)) {
				config_obj->scheme_dir = optarg;
				if (optarg[strlen(optarg) - 1] != '/') {
					config_obj->scheme_dir += '/';
				}
			} else {
				fprintf(stderr, "Cannot access directory %s ", optarg);
				goto exit_failure;
			}
			printf("Scheme directory set to %s\n", optarg);
			break;
		case 'c':
			if (!access(optarg, R_OK)) {
				config_obj->config_dir = optarg;
				if (optarg[strlen(optarg) - 1] != '/') {
					config_obj->config_dir += '/';
				}
			} else {
				fprintf(stderr, "Cannot access directory %s ", optarg);
				goto exit_failure;
			}
			break;
		case 'x':
			if (!access(optarg, X_OK)) {
				config_obj->pub_script_dir = optarg;
				if (optarg[strlen(optarg) - 1] != '/') {
					config_obj->pub_script_dir += '/';
				}
			} else {
				fprintf(stderr, "Cannot access directory %s ", optarg);
				goto exit_failure;
			}
			break;
		case 'y':
			if (!access(optarg, X_OK)) {
				config_obj->script_dir = optarg;
				if (optarg[strlen(optarg) - 1] != '/') {
					config_obj->script_dir += '/';
				}
			} else {
				fprintf(stderr, "Cannot access directory %s ", optarg);
				goto exit_failure;
			}
			break;
		case 'f':
			if (acpi_apm == ACPI) {
				if (!access(optarg, W_OK)) {
					ACPI_Interface::ACPI_EVENTFILE = optarg;
				} else {
					fprintf(stderr, "Cannot access '%s'\n", optarg);
					goto exit_failure;
				}
			}
			break;
		case 'v':
			verbose = strtol(optarg, endptr, 10);
			if (**endptr != '\0' || verbose < 0 || verbose > 31) {
				fprintf(stderr, "Wrong verbose paramater, must be between 0-31\n");
				goto exit_failure;
			} else {
				setDebugLevel(verbose);
			}
			break;
		case 'd':
			daemon = 1;
			break;
		case 'h':
		default:
			usage();
			goto exit_failure;
		}
	}
	return;
      exit_failure:
	delete config_obj;
	config_obj = NULL;
	exit(EXIT_FAILURE);
}

int main(int argc, char **argv)
{
	// vars for get_args:
	int daemon;
	struct sigaction signal_action;
	int acpi_apm;
	FILE *pid_file;

	/* check perms */
	if (geteuid() != 0) {
		usage();
		fprintf(stderr, "Must be run as root\n");
		exit(EXIT_FAILURE);
	}
	// check whether there is already a pid file!
	if (!access(POWERSAVED_PID_FILE_PATH, F_OK)) {
		ifstream is(POWERSAVED_PID_FILE_PATH);
		pid_t pid = 0;
		if (is.good())
			is >> pid;
		if (pid) {
			// finding out if the pid exists or not
			// if it's already taken by some other process we fall
			// back to the normal behaviour: assuming it's a powersaved
			if (kill(pid, 0) == -1 && errno == ESRCH)
				pid = 0;
		}
		if (pid) {
			fprintf(stderr, "powersave daemon is already running. "
				"If it is not running, delete pid file: %s\n", POWERSAVED_PID_FILE_PATH);
			exit(EXIT_FAILURE);
		}
	}
	config_obj = new GeneralConfig();
	if (config_obj == NULL)
		goto exit_failure;

	// check here, we need to know if we can ignore "-f $acpid_socket" in get_args
	acpi_apm = checkACPI();

	get_args(argc, argv, config_obj, daemon, acpi_apm);

	if (daemon)
		daemonize();
	
	if (config_obj->readConfFiles() <= 0) {
		pDebug(DBG_ERR, "Error while parsing config file.");
		printf("Error while parsing config file. See syslog file.");
		goto exit_failure;
	}
	//printf("event: %s, general: %s", event_conf_file.c_str(), general_conf_file.c_str());
	// create pid file here, so I can call exit in get_args(don't need to unlink pid file)
	/*** write process ID file! *************************************/
	if ((pid_file = fopen(POWERSAVED_PID_FILE_PATH, "w")) == NULL) {
		fprintf(stderr, "Could not create pid file: %s, error: %s\n", POWERSAVED_PID_FILE_PATH,
			strerror(errno));
		goto exit_failure;
	}
	fprintf(pid_file, "%d", getpid());
	fclose(pid_file);
	/*** write process ID file! *************************************/

	/***** S E T   S I G N A L   H A N D L I N G ********************/
	// set signal handlers for SIGHUP and exit signals
	// ignore childs, so that main process can go on with listening
	// on socket and hardware event file

	memset(&signal_action, 0, sizeof(signal_action));
	sigaddset(&signal_action.sa_mask, SIGHUP);
	sigaddset(&signal_action.sa_mask, SIGCHLD);
	sigaddset(&signal_action.sa_mask, SIGTERM);
	sigaddset(&signal_action.sa_mask, SIGINT);
	sigaddset(&signal_action.sa_mask, SIGQUIT);
	sigaddset(&signal_action.sa_mask, SIGUSR1);
	sigaddset(&signal_action.sa_mask, SIGPIPE);

	// ignore all signals during signal handling of these signals
	// restart system calls if signal happen
	// this is for restarting select
	signal_action.sa_flags = SA_RESTART || SA_NOCLDSTOP;

	//do a broken pipe from clients
	signal_action.sa_handler = SIG_IGN;
	sigaction(SIGPIPE, &signal_action, 0);

	// re-read configs
	// set the HUP handler to our function
	signal_action.sa_handler = hup_handler;
	sigaction(SIGHUP, &signal_action, 0);

	// terminate properly ...
	// set the TERM handler to our function
	// ToDo: could the signal_action struct
	// be modified here? -> think no
	signal_action.sa_handler = exit_handler;
	sigaction(SIGTERM, &signal_action, 0);
	// set the QUIT handler to our function
	sigaction(SIGQUIT, &signal_action, 0);
	// set the INT handler to our function
	sigaction(SIGINT, &signal_action, 0);

	/* used to inform main process that resume of sleep state is finished */
	signal_action.sa_handler = USR1_handler;
	sigaction(SIGUSR1, &signal_action, 0);

	signal_action.sa_handler = SIG_IGN;
	sigaction(SIGCHLD, &signal_action, 0);
	/****************************************************************/

	// create main loop object _after_ initializing pm object
       	main_loop = new MainLoop();

	if (acpi_apm == ACPI) {
		pm = new ACPI_Interface();
	} else if (acpi_apm == APM) {
		pm = new APM_Interface();
	} else {
		// ToDo: Create a dummy object for no ACPI, no APM case
		pm = new APM_Interface();
		fprintf(stderr, "Neither ACPI, nor APM is supported\n");
//		goto exit_failure;
	}
	cpufreq = new CpufreqManagement();
	pm->start();

	main_loop->run();

exit_failure:
	delete pm;
	delete cpufreq;
	delete config_obj;
	delete main_loop;
	unlink(POWERSAVED_PID_FILE_PATH);
	exit(EXIT_FAILURE);
}


/** @mainpage The Big Picture
 *
 * @image html powersave_architecture.jpeg
 */
