/***************************************************************************
 *                                                                         *
 *                         Powersave Daemon                                *
 *                                                                         *
 *          Copyright (C) 2004,2005 SUSE Linux Products GmbH               *
 *                                                                         *
 *               Author(s): Holger Macht <hmacht@suse.de>                  *
 *                                                                         *
 * This program is free software; you can redistribute it and/or modify it *
 * under the terms of the GNU General Public License as published by the   *
 * Free Software Foundation; either version 2 of the License, or (at you   *
 * option) any later version.                                              *
 *                                                                         *
 * This program is distributed in the hope that it will be useful, but     *
 * WITHOUT ANY WARRANTY; without even the implied warranty of              *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU       *
 * General Public License for more details.                                *
 *                                                                         *
 * You should have received a copy of the GNU General Public License along *
 * with this program; if not, write to the Free Software Foundation, Inc., *
 * 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA                  *
 *                                                                         *
 ***************************************************************************/

#ifndef POWERSAVE_EVENT_H
#define POWERSAVE_EVENT_H

#include <list>
#include <string>
#include <ext/slist>

#include "config_pm.h"

using namespace std;

/** @brief class representing a powersave event like suspend2disk */
class Event {
public:
	/** @brief constructor */
	 Event();

	/** @brief copy constructor */
	 Event(const string &n);

	/** @brief copy constructor */
	 Event(const Event &);

	/** @brief copy constructor */
	 Event & operator=(const Event &);

	/** @brief destructor */
	~Event();

	/** @brief execute only one external program in the list of pending events 
	 * 
	 * I need a whole PM_Interface object here 
	 * to get the current scheme name as a parameter for the invoked script
	 *  and for internal actions, maybe this could be solved nicer ...
	 */
	bool execute_program(string acpi_event_line = "");

	/** @brief set the actions to be executed by this event
	 *
	 * @param actions slist containing the actions
	 *
	 * @return the execute variable as it used in config files.
	 */
	void setActions2Execute(std::list< std::string > actions);

	/** @brief set an action to be executed by this event
	 *
	 * @param action the action to be executed by this event
	 */
	void setActions2Execute(const string &action);

	/** @brief checks whether the given action is internal
	 *
	 * @return integer value with result of the check
	 * @retval 1 if action is an internal action
	 * @retval -1 if not
	 */
	bool isInternalAction();

	// the following is needed for managing events--------------------

	// all the following functions are for setting and getting the different
	// properties

	/** @brief sets the timeout of an event */
	void setTimeout(unsigned long);

	/** @brief gets the timeout of an event */
	unsigned long getTimeout();

	/** @brief starts the timer */
	void start();

	/** @brief sets the id of the event */
	void setID(int id);

	/** @brief returns the event id */
	int ID();

	/** @brief returns the event name
	 *
	 * @retval name of the event
	 */
	string name();

	/** @brief returns the amount of time an event is running already */
	unsigned long timeElapsed();

	/** @brief sets up the names of all internal actions */
	void setupInternalActions();

	list < string >::iterator end();

	/** @brief inserts an action into an event
	 *
	 * @param actions the list which should be adjusted
	 * @param before true if the actions should be inserted before the
	 *        match
	 * @param match the string to match where to insert the action
	 * @param action the action to insert
	 */
	void insertAction(std::list< std::string > &actions, bool before,
			  const std::string &match, const std::string &action);

	/** @brief an iterator which points to the current script in programs_2_execute */
	std::list< string >::iterator _current_program_2_execute;

	/** @brief giving the nicefactor with which they are acecuted
	 *
	 * scripts are executed with negativ nice factor (higher
	 * priority) to avoid race conditions and provide fast
	 * suspend/standby maybe these should be configurable somewhen ...
	 */
	static const int _proxy_nice_factor = -20;

	std::string currentAction();

	/** @brief standard list containing all programs executed for an event
	 * 
	 * this list contains all actions which should get executed for a
	 * specific even. It can contain scripts or internal actions */
	std::list< string > _programs_2_execute;

#ifdef CREATE_DEBUG_OUTPUT
	friend ostream & operator<<(ostream &, const Event &);
#endif

private:
	/** @brief the unique name of an event */
	string _name;

	/** @brief a unique id, assigned by EventManagement */
	int _id;

	/** @brief the timeout in seconds before the event is considered as failed */
	time_t _timeout;

	/** @brief the starttime */
	long _starttime;

	/** @brief the time how long we slept while this event is running */
	unsigned long _sleep_time;

	/** @brief list of internal actions (e.g. throttle) */
	std::list < string > _internalActions;
};

#endif // POWERSAVE_EVENT_H
