/***************************************************************************
 *                                                                         *
 *                         Powersave Daemon                                *
 *                                                                         *
 *          Copyright (C) 2004,2005,2006 SUSE Linux Products GmbH          *
 *                                                                         *
 *               Author(s): Holger Macht <hmacht@suse.de>                  *
 *                                                                         *
 * This program is free software; you can redistribute it and/or modify it *
 * under the terms of the GNU General Public License as published by the   *
 * Free Software Foundation; either version 2 of the License, or (at you   *
 * option) any later version.                                              *
 *                                                                         *
 * This program is distributed in the hope that it will be useful, but     *
 * WITHOUT ANY WARRANTY; without even the implied warranty of              *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU       *
 * General Public License for more details.                                *
 *                                                                         *
 * You should have received a copy of the GNU General Public License along *
 * with this program; if not, write to the Free Software Foundation, Inc., *
 * 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA                  *
 *                                                                         *
 ***************************************************************************/

#include "cpufreq_management.h"
#include "stringutil.h"
#include "globals.h"
#include "cpu.h"
#include "dbus_server.h"

#include <liblazy.h>

using namespace Powersave::Globals;

CpufreqManagement::CpufreqManagement() : _is_supported(false)
{
	if (!config_obj->current_scheme->CPUFREQ_ENABLED) {
		pDebug(DBG_INFO, "cpufrequency scaling is disabled in configuration file");
		return;
	}

	_initial_cpu_count = CPU::enableAll();

	char **capabilities;
	liblazy_hal_find_device_by_capability("cpufreq_control", &capabilities);

	if (capabilities != NULL && capabilities[0] != NULL) {
		_is_supported = true;
		_hal_device = capabilities[0];
	} else
		pDebug(DBG_WARN, "No capability cpufreq_control");
}

CpufreqManagement::~CpufreqManagement()
{
}

void CpufreqManagement::checkCPUHotplug()
{
	bool cpu_enabled = false;

	for (int i = _initial_cpu_count - 1; i > 0; --i) {
		CPU h(i);
		if (i >= config_obj->current_scheme->MAX_CPUS_ONLINE
		    && config_obj->current_scheme->MAX_CPUS_ONLINE != 0) {
			h.disable();
		} else {
			int ret = h.enable();
			if (ret > 0)
				cpu_enabled = true;
		}
	}

	if (cpu_enabled)
		setModes(config_obj->current_scheme->CPUFREQUENCY, NULL);
}

bool CpufreqManagement::isSupported()
{
	return _is_supported;
}

bool CpufreqManagement::setGovernor(char *governor)
{
	return DBus_Server::sendMethodCallAndBlock(DBUS_HAL_INTERFACE,
						   _hal_device.c_str(),
						   DBUS_HAL_CPUFREQ_INTERFACE,
						   "SetCPUFreqGovernor",
						   DBUS_TYPE_STRING,
						   &governor,
						   DBUS_TYPE_INVALID);
}

bool CpufreqManagement::setPerformance(int performance)
{
	return DBus_Server::sendMethodCallAndBlock(DBUS_HAL_INTERFACE,
						   _hal_device.c_str(),
						   DBUS_HAL_CPUFREQ_INTERFACE,
						   "SetCPUFreqPerformance",
						   DBUS_TYPE_INT32,
						   &performance,
						   DBUS_TYPE_INVALID);
}

bool CpufreqManagement::setConsiderNice(int consider)
{
	return DBus_Server::sendMethodCallAndBlock(DBUS_HAL_INTERFACE,
						   _hal_device.c_str(),
						   DBUS_HAL_CPUFREQ_INTERFACE,
						   "SetCPUFreqConsiderNice",
						   DBUS_TYPE_BOOLEAN,
						   &consider,
						   DBUS_TYPE_INVALID);
}

bool CpufreqManagement::setModes(CPUFREQ_MODE mode, EventManagement *eM)
{
	if (DBus_Server::haveClient())
		return true;
	std::string governor;

	if (!_is_supported)
		return false;

	if (eM != NULL) {
		switch (mode) {
		case _DYNAMIC:
			eM->executeEvent("processor.dynamic");		
			break;
		case _PERFORMANCE:
			eM->executeEvent("processor.performance");
			governor.append("performance");
			break;
		case _POWERSAVE:
			eM->executeEvent("processor.powersave");
			break;
		}
	}

	switch (mode) {
	case _DYNAMIC: {
		if (config_obj->current_scheme->CPUFREQ_CONTROL == CPUFREQ_KERNEL)
			governor.append("ondemand");
		else if (config_obj->current_scheme->CPUFREQ_CONTROL == CPUFREQ_USERSPACE)
			governor.append("userspace");
		break;
	}
	case _PERFORMANCE:
		governor.append("performance");
		break;
		case _POWERSAVE:
			governor.append("powersave");
			break;
	}

	
	if (!setGovernor((char*)governor.c_str())) {
		pDebug(DBG_ERR, "Could not set governor '%s'", governor.c_str());
		if (governor == "ondemand") {
			pDebug(DBG_WARN, "Kernel ondemand governor not working, "
			       "trying userspace instead");
		
			if (!setGovernor("userspace")) {
				pDebug(DBG_ERR, "Neither ondemand nor userspace governor working");
				return false;
			}
		}
	}

	if (mode == _DYNAMIC) {
		if (!setPerformance(config_obj->current_scheme->CPUFREQ_DYNAMIC_PERFORMANCE))
			pDebug(DBG_WARN, "Could not set dynamic performance setting");
		if (!setConsiderNice(config_obj->current_scheme->CONSIDER_NICE))
			pDebug(DBG_WARN, "Could not set consider nice setting");
	}
	return true;
}

char *CpufreqManagement::getGovernor()
{
	char *governor = NULL;
	if (!DBus_Server::sendMethodCallWithReturn(DBUS_HAL_INTERFACE,
						   _hal_device.c_str(),
						   DBUS_HAL_CPUFREQ_INTERFACE,
						   "GetCPUFreqGovernor",
						   DBUS_TYPE_STRING,
						   &governor)) {
		return NULL;
	}
	return governor;
}

CPUFREQ_MODE CpufreqManagement::getMode()
{
	CPUFREQ_MODE mode = _DYNAMIC;
	char *governor = getGovernor();
	if (governor == NULL)
		return mode;

	if (!strcmp(governor, "ondemand") || !strcmp(governor, "userspace"))
		mode = _DYNAMIC;
	else if (!strcmp(governor, "powersave"))
		mode = _POWERSAVE;
	else if (!strcmp(governor, "performance"))
		mode = _PERFORMANCE;

	return mode;
}

CPUFREQ_CONTROL_MODE CpufreqManagement::controlMode()
{
	if (!_is_supported)
		return CPUFREQ_NONE;

	char *governor =  getGovernor();
	if (governor == NULL)
		return CPUFREQ_NONE;
	CPUFREQ_CONTROL_MODE control = CPUFREQ_KERNEL;

	if (!strcmp(governor, "ondemand"))
		control = CPUFREQ_KERNEL;
	else if (!strcmp(governor, "userspace"))
		control = CPUFREQ_USERSPACE;

	return control;
}

int CpufreqManagement::enableCPU(int cpu)
{
	CPU c(cpu);

	if (!c.hotpluggable())
		return -1;

	int ret = c.enable();
	if (ret > 0)
		setModes(config_obj->current_scheme->CPUFREQUENCY, NULL);
	return ret;
}

int CpufreqManagement::disableCPU(int cpu)
{
	// we cannot disable the first CPU, it runs the main timer
	if (cpu == 0)
		return -1;

	CPU c(cpu);

	if (!c.hotpluggable())
		return -1;

	return c.disable();
}
