#include "cpu.h"
#include "powerlib.h"

#include <sstream>
#include <errno.h>
#include <stdarg.h>

#define SYSFS_FILES "/sys/devices/system/cpu/"

using namespace std;

CPU::CPU(int id) : _cpu_base(id)
{
	ostringstream strstr;
	strstr << SYSFS_FILES << "cpu" << _cpu_base << "/online";
	ONLINE_FILE = strstr.str();
}

int CPU::enable()
{
	if (online())
		return 0;

	pDebug(DBG_INFO, "Trying to enable CPU %d", _cpu_base);
	if (!write_line(ONLINE_FILE.c_str(), "1")) {
		pDebug(DBG_DEBUG, "Unable to open file: %s\n", ONLINE_FILE.c_str());
		return -1;
	}
	
	pDebug(DBG_INFO, "Set CPU %d online", _cpu_base);

	if (!online()) {
		pDebug(DBG_WARN, "Onlined CPU, but request was not accepted: %s\n",
		       ONLINE_FILE.c_str());
		return -1;
	}

	return 1;
}

int CPU::disable()
{
	if (!online())
		return 0;

	if (_cpu_base == 0) {
		pDebug(DBG_DEBUG, "CPU 0 can't be offlined.");
		return -1;
	}
	
	pDebug(DBG_INFO, "Trying to disable CPU %d", _cpu_base);
	if (!write_line(ONLINE_FILE.c_str(), "0")) {
		pDebug(DBG_DEBUG, "Unable to open file: %s\n",
		       ONLINE_FILE.c_str());
		return -1;
	}

	pDebug(DBG_INFO, "Set CPU %d offline", _cpu_base);
	if (online()) {
		pDebug(DBG_WARN, "Offlined CPU, but request was not accepted: %s\n",
		       ONLINE_FILE.c_str());
		return -1;
	}

	return 1;
}

bool CPU::online()
{
	if (!hotpluggable())
		return true;

	char online_str[2];

	if (!read_line(ONLINE_FILE.c_str(), online_str, 2)) {
		pDebug(DBG_DEBUG, "Unable to open file: %s\n", ONLINE_FILE.c_str());
		return false;
	}
	
	int online = atoi(online_str);

	if (!online)
		return false;
	
	return true;
}

bool CPU::hotpluggable()
{
	if (access(ONLINE_FILE.c_str(), F_OK) < 0)
		return false;
	return true;
}

int CPU::enableAll()
{
	// start with CPU 1, because CPU 0 is always online atm
	int i = 1;

	while (true) {
		CPU cpu(i);
		if (!cpu.hotpluggable())
			break;
		cpu.enable();
		i++;
	} 

	pDebug(DBG_DEBUG, "Found %d hotpluggable CPUs in system", i);
	return i;
}


bool CPU::read_line(const char *filename, char *line, unsigned len)
{
	FILE *fp = fopen(filename, "r");
	if (!fp) {
		pDebug(DBG_ERR, "Could not open '%s': %s", filename, strerror(errno));
		return false;
	}
	if ((!fgets(line, len, fp))) {
		pDebug(DBG_ERR, "Could not read from '%s': %s", filename, strerror(errno));
		fclose(fp);
		return false;
	}
	fclose(fp);
	return true;
}

bool CPU::write_line(const char *filename, const char *fmt, ...)
{
	FILE *fp = fopen(filename, "w+");
	if (!fp) {
		pDebug(DBG_WARN, "Could not open file for writing: %s; %s", filename, strerror(errno));
		return false;
	}
	va_list ap;
	va_start(ap, fmt);	// get variable argument list passed
	if (vfprintf(fp, fmt, ap) < 0) {
		pDebug(DBG_WARN, "Could not write to file: %s", filename);
		fclose(fp);
		return false;
	}
	va_end(ap);
	fclose(fp);
	return true;
}

