/***************************************************************************
 *                                                                         *
 *                         Powersave Daemon                                *
 *                                                                         *
 *          Copyright (C) 2004,2005,2006 SUSE Linux Products GmbH          *
 *                                                                         *
 *               Author(s): Timo Hoenig <thoenig@suse.de>                  *
 *                          Danny Kukawka <danny.kukawka@web.de>           *
 *                                                                         *
 * This program is free software; you can redistribute it and/or modify it *
 * under the terms of the GNU General Public License as published by the   *
 * Free Software Foundation; either version 2 of the License, or (at you   *
 * option) any later version.                                              *
 *                                                                         *
 * This program is distributed in the hope that it will be useful, but     *
 * WITHOUT ANY WARRANTY; without even the implied warranty of              *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU       *
 * General Public License for more details.                                *
 *                                                                         *
 * You should have received a copy of the GNU General Public License along *
 * with this program; if not, write to the Free Software Foundation, Inc., *
 * 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA                  *
 *                                                                         *
 ***************************************************************************/

#include "brightness.h"
#include "dbus_server.h"
#include "globals.h"
#include "powerlib.h"

#include <liblazy.h>

Brightness::Brightness() : _supported(false), _last_percent(-1)
{
	char **panels;

	liblazy_hal_find_device_by_capability("laptop_panel", &panels);

	if (panels == NULL || panels[0] == NULL) {
		pDebug(DBG_INFO, "No Laptop Panels found\n");
		return;
	}

	pDebug(DBG_INFO, "Found panel devices with udi %s", panels[0]);

	_udi = panels[0];
	liblazy_free_strlist(panels);

	char **interfaces;
	liblazy_hal_get_property_strlist((char*)_udi.c_str(), "info.interfaces", &interfaces);
	
	if (interfaces != NULL) {
		_interface = interfaces[0];
		liblazy_free_strlist(interfaces);
	} else {
		pDebug(DBG_WARN, "Could not get Laptop Panel interface");
		return;
	}

	pDebug(DBG_INFO, "found interface '%s' and udi '%s'\n",
	       _interface.c_str(), _udi.c_str());

	_supported = true;
}

Brightness::~Brightness()
{
}

int Brightness::Get()
{
	unsigned int brightness = -1;

	if (!_supported)
		return -1;

	if (!DBus_Server::sendMethodCallWithReturn("org.freedesktop.Hal",
						   _udi.c_str(),
						   "org.freedesktop.Hal.Device.LaptopPanel",
						   "GetBrightness",
						   DBUS_TYPE_UINT32,
						   &brightness))
		pDebug(DBG_INFO, "Could not get current brightness from HAL");

	return (int)brightness;
}

void Brightness::Set(int level)
{
	if (!_supported)
		return;

	int num_levels = GetLevels();

	if (level > num_levels - 1)
		level = num_levels - 1;
	else if (level < 0)
		level = 0;

	if (!DBus_Server::sendMethodCall(DBUS_HAL_INTERFACE,
					 _udi.c_str(),
					 _interface.c_str(),
					 "SetBrightness",
					 DBUS_TYPE_INT32,
					 &level,
					 DBUS_TYPE_INVALID))
		pDebug(DBG_INFO, "Could not set brightness");
}

void Brightness::Min()
{
	Set(0);
	return;
}

void Brightness::Med()
{
	Set(GetLevels() / 2);
	return;
}

void Brightness::Max()
{
	Set(GetLevels() - 1);
	return;
}

void Brightness::Up()
{
	Set(Get() + 1);
	return;
}

void Brightness::Down()
{
	Set(Get() - 1);
	return;
}

int Brightness::GetLevels()
{
	if (!_supported)
		return -1;

	char *prop = "laptop_panel.num_levels";
	int num_levels;
	liblazy_hal_get_property_int((char*)_udi.c_str(), prop, &num_levels);

	return num_levels;
}

// a helper function: give percent, get level
int Brightness::PercentToLevel(int percent)
{
	if (!_supported)
		return -1;

	int b, l, comp;
	l = GetLevels();
	comp = 50 / l;	// (100/l)/2, compensation
	if (percent < 0)
		percent = 0;
	if (percent > 100)
		percent = 100;

	b = l * (percent + comp) / 100;
	return b;
}

void Brightness::SetPercent(int percent)
{
	Set(PercentToLevel(percent));
	_last_percent = percent;
}

int Brightness::GetPercent()
{
	if (!_supported)
		return -1;

	int b, l;
	b = Get();
	l = GetLevels() - 1;
	
	if ((b < 0) || (l <= 0))
		return -1;

	if ((_last_percent >= 0) && (PercentToLevel(_last_percent) == b))
		return _last_percent;

	_last_percent = ((b * 1000 + 5) / l) / 10;
	return _last_percent;
}
