/* SPDX-License-Identifier: GPL-2.0+ */
/*
 * Internal header file for scenes
 *
 * Copyright 2022 Google LLC
 * Written by Simon Glass <sjg@chromium.org>
 */

#ifndef __SCENE_INTERNAL_H
#define __SCENE_INTERNAL_H

#include <linux/types.h>

struct expo;
struct expo_action;
struct expo_arrange_info;
struct expo_theme;
struct scene_obj;
struct scene_obj_menu;
struct scene_obj_textline;
struct scene_obj_txtedit;
struct scene_txt_generic;
struct vidconsole_bbox;

enum scene_obj_t;

typedef int (*expo_scene_obj_iterator)(struct scene_obj *obj, void *priv);

/**
 * enum scene_bbox_t - Parts of an object which can have a bounding box
 *
 * Objects can provide any or all of these bounding boxes
 *
 * @SCENEBB_label: Menu-item label
 * @SCENEBB_key: Menu-item key label
 * @SCENEBB_desc: Menu-item Description
 * @SCENEBB_curitem: Current item (pointed to)
 * @SCENEBB_all: All the above objects combined
 */
enum scene_bbox_t {
	SCENEBB_label,
	SCENEBB_key,
	SCENEBB_desc,
	SCENEBB_curitem,
	SCENEBB_all,

	SCENEBB_count,
};

/**
 * expo_lookup_scene_id() - Look up a scene ID
 *
 * @exp: Expo to use
 * @id: scene ID to look up
 * Returns: Scene for that ID, or NULL if none
 */
struct scene *expo_lookup_scene_id(struct expo *exp, uint scene_id);

/**
 * resolve_id() - Automatically allocate an ID if needed
 *
 * @exp: Expo to use
 * @id: ID to use, or 0 to auto-allocate one
 * Returns: Either @id, or the auto-allocated ID
 */
uint resolve_id(struct expo *exp, uint id);

/**
 * scene_obj_find() - Find an object in a scene
 *
 * Note that @type is used to restrict the search when the object type is known.
 * If any type is acceptable, set @type to SCENEOBJT_NONE
 *
 * @scn: Scene to search
 * @id: ID of object to find
 * @type: Type of the object, or SCENEOBJT_NONE to match any type
 * Returns: Object found, or NULL if not found
 */
void *scene_obj_find(const struct scene *scn, uint id, enum scene_obj_t type);

/**
 * scene_obj_find_by_name() - Find an object in a scene by name
 *
 * @scn: Scene to search
 * @name: Name to search for
 */
void *scene_obj_find_by_name(struct scene *scn, const char *name);

/**
 * scene_obj_add() - Add a new object to a scene
 *
 * @scn: Scene to update
 * @name: Name to use (this is allocated by this call)
 * @id: ID to use for the new object (0 to allocate one)
 * @type: Type of object to add
 * @size: Size to allocate for the object, in bytes
 * @objp: Returns a pointer to the new object (must not be NULL)
 * Returns: ID number for the object (generally @id), or -ve on error
 */
int scene_obj_add(struct scene *scn, const char *name, uint id,
		  enum scene_obj_t type, uint size, struct scene_obj **objp);

/**
 * scene_obj_flag_clrset() - Adjust object flags
 *
 * @scn: Scene to update
 * @id: ID of object to update
 * @clr: Bits to clear in the object's flags
 * @set: Bits to set in the object's flags
 * Returns 0 if OK, -ENOENT if the object was not found
 */
int scene_obj_flag_clrset(struct scene *scn, uint id, uint clr, uint set);

/**
 * scene_calc_dims() - Calculate the dimensions of the scene objects
 *
 * Updates the width and height of all objects based on their contents
 *
 * @scn: Scene to update
 * @do_menus: true to calculate only menus, false to calculate everything else
 * Returns 0 if OK, -ENOTSUPP if there is no graphical console
 */
int scene_calc_dims(struct scene *scn, bool do_menus);

/**
 * scene_menu_arrange() - Set the position of things in the menu
 *
 * This updates any items associated with a menu to make sure they are
 * positioned correctly relative to the menu. It also selects the first item
 * if not already done
 *
 * @scn: Scene to update
 * @arr: Arrangement information
 * @menu: Menu to process
 * Returns: 0 if OK, -ve on error
 */
int scene_menu_arrange(struct scene *scn, struct expo_arrange_info *arr,
		       struct scene_obj_menu *menu);

/**
 * scene_textline_arrange() - Set the position of things in a textline
 *
 * This updates any items associated with a textline to make sure they are
 * positioned correctly relative to the textline.
 *
 * @scn: Scene to update
 * @arr: Arrangement information
 * @tline: textline to process
 * Returns: 0 if OK, -ve on error
 */
int scene_textline_arrange(struct scene *scn, struct expo_arrange_info *arr,
			   struct scene_obj_textline *tline);

/**
 * scene_apply_theme() - Apply a theme to a scene
 *
 * @scn: Scene to update
 * @theme: Theme to apply
 * Returns: 0 if OK, -ve on error
 */
int scene_apply_theme(struct scene *scn, struct expo_theme *theme);

/**
 * scene_menu_send_key() - Send a key to a menu for processing
 *
 * @scn: Scene to use
 * @menu: Menu to use
 * @key: Key code to send (KEY_...)
 * @event: Place to put any event which is generated by the key
 * Returns: 0 if OK, -ENOTTY if there is no current menu item, other -ve on other
 *	error
 */
int scene_menu_send_key(struct scene *scn, struct scene_obj_menu *menu, int key,
			struct expo_action *event);

/**
 * scene_textline_send_key() - Send a key to a textline for processing
 *
 * @scn: Scene to use
 * @tline: textline to use
 * @key: Key code to send (KEY_...)
 * @event: Place to put any event which is generated by the key
 * Returns: 0 if OK (always)
 */
int scene_textline_send_key(struct scene *scn, struct scene_obj_textline *tline,
			    int key, struct expo_action *event);

/**
 * scene_menu_destroy() - Destroy a menu in a scene
 *
 * @scn: Scene to destroy
 */
void scene_menu_destroy(struct scene_obj_menu *menu);

/**
 * scene_menu_display() - Display a menu as text
 *
 * @menu: Menu to display
 * Returns: 0 if OK, -ENOENT if @id is invalid
 */
int scene_menu_display(struct scene_obj_menu *menu);

/**
 * scene_destroy() - Destroy a scene and all its memory
 *
 * @scn: Scene to destroy
 */
void scene_destroy(struct scene *scn);

/**
 * scene_render() - Render a scene
 *
 * This is called from expo_render()
 *
 * @scn: Scene to render
 * Returns: 0 if OK, -ve on error
 */
int scene_render(struct scene *scn);

/**
 * scene_send_key() - set a keypress to a scene
 *
 * @scn: Scene to receive the key
 * @key: Key to send (KEYCODE_UP)
 * @event: Returns resulting event from this keypress
 * Returns: 0 if OK, -ve on error
 */
int scene_send_key(struct scene *scn, int key, struct expo_action *event);

/**
 * scene_render_deps() - Render an object and its dependencies
 *
 * @scn: Scene to render
 * @id: Object ID to render (or 0 for none)
 * Returns: 0 if OK, -ve on error
 */
int scene_render_deps(struct scene *scn, uint id);

/**
 * scene_menu_render_deps() - Render a menu and its dependencies
 *
 * Renders the menu and all of its attached objects
 *
 * @scn: Scene to render
 * @menu: Menu to render
 * Returns: 0 if OK, -ve on error
 */
int scene_menu_render_deps(struct scene *scn, struct scene_obj_menu *menu);

/**
 * scene_textline_render_deps() - Render a textline and its dependencies
 *
 * Renders the textline and all of its attached objects
 *
 * @scn: Scene to render
 * @tline: textline to render
 * Returns: 0 if OK, -ve on error
 */
int scene_textline_render_deps(struct scene *scn,
			       struct scene_obj_textline *tline);

/**
 * scene_menu_calc_dims() - Calculate the dimensions of a menu
 *
 * Updates the width and height of the menu based on its contents
 *
 * @menu: Menu to update
 * Returns 0 if OK, -ENOTSUPP if there is no graphical console
 */
int scene_menu_calc_dims(struct scene_obj_menu *menu);

/**
 * scene_iter_objs() - Iterate through all scene objects
 *
 * @scn: Scene to process
 * @iter: Iterator to call on each object
 * @priv: Private data to pass to the iterator, in addition to the object
 * Return: 0 if OK, -ve on error
 */
int scene_iter_objs(struct scene *scn, expo_scene_obj_iterator iter,
		    void *priv);

/**
 * expo_iter_scene_objects() - Iterate through all scene objects
 *
 * @exp: Expo to process
 * @iter: Iterator to call on each object
 * @priv: Private data to pass to the iterator, in addition to the object
 * Return: 0 if OK, -ve on error
 */
int expo_iter_scene_objs(struct expo *exp, expo_scene_obj_iterator iter,
			 void *priv);

/**
 * scene_menuitem_find() - Find the menu item for an ID
 *
 * Looks up the menu to find the item with the given ID
 *
 * @menu: Menu to check
 * @id: ID to look for
 * Return: Menu item, or NULL if not found
 */
struct scene_menitem *scene_menuitem_find(const struct scene_obj_menu *menu,
					  int id);

/**
 * scene_menuitem_find_seq() - Find the menu item at a sequential position
 *
 * This numbers the items from 0 and returns the seq'th one
 *
 * @menu: Menu to check
 * @seq: Sequence number to look for
 * Return: menu item if found, else NULL
 */
struct scene_menitem *scene_menuitem_find_seq(const struct scene_obj_menu *menu,
					      uint seq);

/**
 * scene_menuitem_find_val() - Find the menu item with a given value
 *
 * @menu: Menu to check
 * @find_val: Value to look for
 * Return: menu item if found, else NULL
 */
struct scene_menitem *scene_menuitem_find_val(const struct scene_obj_menu *menu,
					      int val);

/**
 * scene_bbox_join() - update bouding box with a given src box
 *
 * Updates @dst so that it encompasses the bounding box @src
 *
 * @src: Input bounding box
 * @inset: Amount of inset to use for width
 * @dst: Bounding box to update
 * Return: 0 if OK, -ve on error
 */
int scene_bbox_join(const struct vidconsole_bbox *src, int inset,
		    struct vidconsole_bbox *dst);

/**
 * scene_bbox_union() - update bouding box with the demensions of an object
 *
 * Updates @bbox so that it encompasses the bounding box of object @id
 *
 * @snd: Scene containing object
 * @id: Object id
 * @inset: Amount of inset to use for width
 * @bbox: Bounding box to update
 * Return: 0 if OK, -ve on error
 */
int scene_bbox_union(struct scene *scn, uint id, int inset,
		     struct vidconsole_bbox *bbox);

/**
 * scene_textline_calc_dims() - Calculate the dimensions of a textline
 *
 * Updates the width and height of the textline based on its contents
 *
 * @tline: Textline to update
 * Returns 0 if OK, -ENOTSUPP if there is no graphical console
 */
int scene_textline_calc_dims(struct scene_obj_textline *tline);

/**
 * scene_menu_calc_bbox() - Calculate bounding boxes for the menu
 *
 * @menu: Menu to process
 * @bbox: List of bounding box to fill in
 * Return: 0 if OK, -ve on error
 */
void scene_menu_calc_bbox(struct scene_obj_menu *menu,
			  struct vidconsole_bbox *bbox);

/**
 * scene_textline_calc_bbox() - Calculate bounding box for the textline
 *
 * @textline: Menu to process
 * @bbox: Returns bounding box of textline including prompt
 * @edit_bbox: Returns bounding box of editable part
 * Return: 0 if OK, -ve on error
 */
void scene_textline_calc_bbox(struct scene_obj_textline *menu,
			      struct vidconsole_bbox *bbox,
			      struct vidconsole_bbox *label_bbox);

/**
 * scene_obj_calc_bbox() - Calculate bounding boxes for an object
 *
 * @obj: Object to process
 * @bbox: Returns bounding boxes for object
 * Return: 0 if OK, -ve on error
 */
int scene_obj_calc_bbox(struct scene_obj *obj, struct vidconsole_bbox *bbox);

/**
 * scene_textline_open() - Open a textline object
 *
 * Set up the text editor ready for use
 *
 * @scn: Scene containing the textline
 * @tline: textline object
 * Return: 0 if OK, -ve on error
 */
int scene_textline_open(struct scene *scn, struct scene_obj_textline *tline);

/**
 * scene_textline_close() - Close a textline object
 *
 * Close out the text editor after use
 *
 * @scn: Scene containing the textline
 * @tline: textline object
 * Return: 0 if OK, -ve on error
 */
int scene_textline_close(struct scene *scn, struct scene_obj_textline *tline);

/**
 * scene_calc_arrange() - Calculate sizes needed to arrange a scene
 *
 * Checks the size of some objects and stores this info to help with a later
 * scene arrangement
 *
 * @scn: Scene to check
 * @arr: Place to put scene-arrangement info
 * Returns: 0 if OK, -ve on error
 */
int scene_calc_arrange(struct scene *scn, struct expo_arrange_info *arr);

/**
 * scene_txt_generic_init() - Set up the generic part of a text object
 *
 * @exp: Expo containing the object
 * @gen: Generic text info
 * @name: Object name
 * @str_id: String ID for the text
 * @str: Initial text string for the object, or NULL to just use str_id
 */
int scene_txt_generic_init(struct expo *exp, struct scene_txt_generic *gen,
			   const char *name, uint str_id, const char *str);

#endif /* __SCENE_INTERNAL_H */
