/* torsmo, a system monitor
 *
 * This program is licensed under BSD license, read COPYING
 *
 * see torsmorc.sample for configuration
 */

#include "torsmo.h"
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <time.h>
#include <signal.h>
#include <unistd.h>
#include <errno.h>
#include <string.h>
#include <limits.h>
#include <sys/time.h>
#include <sys/utsname.h>
#include <sys/stat.h>
#include <X11/Xlib.h>
#include <X11/Xatom.h>
#include <X11/Xutil.h>

/* alignments */
enum {
  TOP_LEFT = 1,
  TOP_RIGHT,
  BOTTOM_LEFT,
  BOTTOM_RIGHT
};

/* default stuff */

/* default config file, undef if you don't want to use this at all */
#define CONFIG_FILE "$HOME/.torsmorc"
static char *current_config;

/* mail spool */
#define MAIL_FILE "$MAIL"
static char *current_mail_spool;

/* set to 1 if you want all text to be in uppercase */
static unsigned int stuff_in_upper_case = 0;

/* Position on the screen */
static int text_alignment = BOTTOM_LEFT;
static int gap_x = 5;
static int gap_y = 5;

/* Font used */
static const char *font = "6x10";

/* Update interval */
static double update_interval = 10.0;

/* fork? */
static int fork_to_background = 0;

/* border */
static int draw_borders = 0;
static int stippled_borders = 0;

static int draw_shades = 1;

static long default_fg_color;
static long default_bg_color;

#ifdef OWN_WINDOW
/* create own window or draw stuff to root? */
static int own_window = 0;
#endif

static int fixed_size = 0;

/* no buffers in used memory? */
int no_buffers = 1;

/* Text that is shown */
static char original_text[] =
"$nodename - $sysname $kernel on $machine\n"
"$hr\n"
"${color grey}Uptime$color $uptime\n"
"${color grey}RAM Usage:$color $mem/$memmax - $memperc% ${membar 4}\n"
"${color grey}Swap Usage:$color $swap/$swapmax - $swapperc% ${swapbar 4}\n"
"${color grey}CPU Usage:$color $cpu% ${cpubar 4}\n"
"${color grey}Processes:$color $processes  ${color grey}Running:$color $running_processes\n"
"$hr\n"
"${color grey}Networking:\n"
" Up:$color ${upspeed eth0} k/s${color grey} - Down:$color ${downspeed eth0} k/s\n"
"${color grey}Temperatures:\n"
" CPU:$color ${i2c temp 1}C${color grey} - MB:$color ${i2c temp 2}C\n"
;

static char *text = original_text;

static Display *display;
static int screen;
static GC gc;
static XFontStruct *font_info;
static int display_width, display_height;

static Window win;
static int window_width, window_height;

#ifdef OWN_WINDOW
static int window_x, window_y;
#endif

/* workarea, this is where window / text is aligned */
static long workarea[4];

/* formatted text to render on screen, generated in generate_text(),
 * drawn in draw_stuff() */

#define TEXT_BUFFER_SIZE (1024*4)

static char text_buffer[TEXT_BUFFER_SIZE];

/* special stuff in text_buffer */

#define SPECIAL_CHAR '\x01'

enum {
  HORIZONTAL_LINE,
  STIPPLED_HR,
  BAR,
  FG,
  BG,
};

static struct special_t {
  int type;
  int height;
  long arg;
} specials[128];

static int special_count;

static void new_bar(char *buf, int h, int usage) {
  buf[0] = SPECIAL_CHAR;
  buf[1] = '\0';

  specials[special_count].type = BAR;
  specials[special_count].height = h;
  specials[special_count].arg = (usage > 255) ? 255 : ((usage < 0) ? 0 : usage);
  special_count++;
}

static inline void new_hr(char *buf, int a) {
  buf[0] = SPECIAL_CHAR;
  buf[1] = '\0';

  specials[special_count].type = HORIZONTAL_LINE;
  specials[special_count].height = a;
  special_count++;
}

static inline void new_stippled_hr(char *buf, int a, int b) {
  buf[0] = SPECIAL_CHAR;
  buf[1] = '\0';

  specials[special_count].type = STIPPLED_HR;
  specials[special_count].height = b;
  specials[special_count].arg = a;
  special_count++;
}

static unsigned long get_color(const char *name) {
  XColor color;

  color.pixel = 0;
  if (!XParseColor(display, DefaultColormap(display, screen), name, &color))
    ERR("can't parse X color '%s'", name);
  else if (!XAllocColor(display, DefaultColormap(display, screen), &color))
    ERR("can't allocate X color '%s'", name);

  return color.pixel;
}

static inline void new_fg(char *buf, long c) {
  buf[0] = SPECIAL_CHAR;
  buf[1] = '\0';

  specials[special_count].type = FG;
  specials[special_count].arg = c;
  special_count++;
}

static inline void new_bg(char *buf, long c) {
  buf[0] = SPECIAL_CHAR;
  buf[1] = '\0';

  specials[special_count].type = BG;
  specials[special_count].arg = c;
  special_count++;
}

/* quite boring functions */

double get_time() {
  struct timeval tv;
  gettimeofday(&tv, 0);
  return tv.tv_sec + tv.tv_usec / 1000000.0;
}

FILE *open_file(const char *file, int *reported) {
  FILE *fp = fopen(file, "r");
  if (!fp) {
    if (!reported || *reported == 0) {
      ERR("can't open %s: %s", file, strerror(errno));
      if (reported) *reported = 1;
    }
    return 0;
  }

  return fp;
}

static void for_each_line(char *b, void (*f)(char *)) {
  char *ps, *pe;

  for (ps=b, pe=b; *pe; pe++) {
    if (*pe == '\n') {
      *pe = '\0';
      f(ps);
      *pe = '\n';
      ps = pe+1;
    }
  }

  if (ps < pe) f(ps);
}

static void convert_escapes(char *buf) {
  char *p = buf, *s = buf;

  while (*s) {
    if(*s == '\\') {
      s++;
      if(*s == 'n') *p++ = '\n';
      else if(*s == '\\') *p++ = '\\';
      s++;
    }
    else
      *p++ = *s++;
  }
  *p = '\0';
}

/* memory info */

unsigned int mem=0, memmax=0, swap=0, swapmax=0, bufmem=0;

/* cpu usage stuff */

double cpu_usage = 0.0;

/* network interface stuff */

static struct net_stat netstats[16];

struct net_stat *get_net_stat(const char *dev) {
  unsigned int i;

  if (!dev) return 0;

  /* find interface stat */
  for (i=0; i<16; i++) {
    if (netstats[i].dev && strcmp(netstats[i].dev, dev) == 0)
      return &netstats[i];
  }

  /* wasn't found? add it */
  if (i == 16) {
    for (i=0; i<16; i++) {
      if (netstats[i].dev == 0) {
        netstats[i].dev = strdup(dev);
        return &netstats[i];
      }
    }
  }

  CRIT_ERR("too many interfaces used (limit is 16)");
  return 0;
}

static int new_mail_count, mail_count;
static time_t last_mail_mtime;
static double last_mail_update;

static void update_mail_count() {
  struct stat buf;

  if (current_mail_spool == NULL) return;

  /* don't check mail so often (9.5s is minimum interval) */
  if (current_update_time - last_mail_update < 9.5)
    return;
  else
    last_mail_update = current_update_time;

  if (stat(current_mail_spool, &buf)) {
    static int rep;
    if (!rep) {
      ERR("can't stat %s: %s", current_mail_spool, strerror(errno));
      rep = 1;
    }
    return;
  }

  if (buf.st_mtime != last_mail_mtime) {
    /* yippee, modification time has changed, let's read mail count! */
    static int rep;
    FILE *fp;
    int reading_status = 0;

    /* could lock here but I don't think it's really worth it because
     * this isn't going to write mail spool */

    new_mail_count = 0;
    mail_count = 0;

    fp = open_file(current_mail_spool, &rep);
    if (!fp) return;

    /* NOTE: adds mail as new if there isn't Status-field at all */

    while (!feof(fp)) {
      char buf[128];
      if (fgets(buf, 128, fp) == NULL) break;

      if (strncmp(buf, "From ", 5) == 0) {
        /* ignore MAILER-DAEMON */
        if (strncmp(buf+5, "MAILER-DAEMON ", 14) != 0) {
          mail_count++;

          if (reading_status)
            new_mail_count++;
          else
            reading_status = 1;
        }
      }
      else {
        if (reading_status && strncmp(buf, "Status:", 7) == 0) {
          /* check that mail isn't already read */
          if (strchr(buf+7, 'R') == NULL)
            new_mail_count++;

          reading_status = 0;
          continue;
        }
      }

      /* skip until \n */
      while (strchr(buf, '\n') == NULL && !feof(fp))
        fgets(buf, 128, fp);
    }

    fclose(fp);

    if (reading_status) new_mail_count++;

    last_mail_mtime = buf.st_mtime;
  }
}

static void human_readable(long long a, char *buf) {
  if (a >= 1024*1024*1024)
    snprintf(buf, 255, "%.2fG", (a/1024/1024)/1024.0);
  else if (a >= 1024*1024) {
    double m = (a/1024)/1024.0;
    if(m >= 100.0)
      snprintf(buf, 255, "%.0fM", m);
    else
      snprintf(buf, 255, "%.1fM", m);
  }
  else if (a >= 1024)
    snprintf(buf, 255, "%Ldk", a/1024L);
  else
    snprintf(buf, 255, "%Ld", a);
}

void format_time(char *buf, unsigned int n, long t) {
  if(t >= 24*60*60)
    snprintf(buf, n, "%ldd %ldh %ldmin", t/60/60/24,
        (t/60/60) % 24,
        (t/60) % 60);
  else if(t >= 60*60)
    snprintf(buf, n, "%ldh %ldmin",
        (t/60/60) % 24,
        (t/60) % 60);
  else
    snprintf(buf, n, "%ldmin %lds", t/60, t % 60);
}

static struct utsname uname_s;

static void update_uname() {
  uname(&uname_s);
}

/* text handling */

static int need_cpu_usage, need_net_stats, need_mails, need_meminfo;

enum text_object_type {
  OBJ_acpitemp,
  OBJ_battery,
  OBJ_color,
  OBJ_cpu,
  OBJ_cpubar,
  OBJ_downspeed,
  OBJ_downspeedf,
  OBJ_pre_exec,
  OBJ_exec,
  OBJ_hr,
  OBJ_i2c,
  OBJ_kernel,
  OBJ_loadavg,
  OBJ_machine,
  OBJ_mails,
  OBJ_mem,
  OBJ_membar,
  OBJ_memmax,
  OBJ_memperc,
  OBJ_new_mails,
  OBJ_nodename,
  OBJ_processes,
  OBJ_running_processes,
  OBJ_shadecolor,
  OBJ_stippled_hr,
  OBJ_swap,
  OBJ_swapbar,
  OBJ_swapmax,
  OBJ_swapperc,
  OBJ_sysname,
  OBJ_text,
  OBJ_temp1, /* i2c is used instead in these */
  OBJ_temp2,
  OBJ_time,
  OBJ_totaldown,
  OBJ_totalup,
  OBJ_updates,
  OBJ_upspeed,
  OBJ_upspeedf,
  OBJ_uptime
};

struct text_object {
  int type;
  union {
    char *s;
    int i;
    long l;
    struct net_stat *net;
    struct {
      int fd;
      char *dev;
      char *type;
      int n;
    } i2c;
    struct {
      int a, b;
    } pair;
    unsigned char loadavg[3];
  } data;
};

static unsigned int text_object_count;
static struct text_object *text_objects;

static struct text_object *text_object_new() {
  text_object_count++;
  text_objects = (struct text_object *) realloc(text_objects,
      sizeof(struct text_object) * text_object_count);
  memset(&text_objects[text_object_count-1], 0, sizeof(struct text_object));

  return &text_objects[text_object_count-1];
}

static void free_text_objects() {
  unsigned int i;
  for (i=0; i<text_object_count; i++) {
    switch (text_objects[i].type) {
    case OBJ_time:
    case OBJ_text:
    case OBJ_exec:
    case OBJ_pre_exec:
      free(text_objects[i].data.s);
      break;

    case OBJ_i2c:
      close(text_objects[i].data.i2c.fd);
      free(text_objects[i].data.i2c.dev);
      free(text_objects[i].data.i2c.type);
      break;
    }
  }

  free(text_objects);
  text_objects = NULL;
  text_object_count = 0;
}

static void construct_text_object(const char *s, const char *arg) {
  struct text_object *obj = text_object_new();

#define OBJ(a) if (strcmp(s, #a) == 0) { obj->type = OBJ_##a; {
#define END ; } } else

  if (s[0] == '#') {
    obj->type = OBJ_color;
    obj->data.l = get_color(s);
  }
  else
  OBJ(acpitemp)
  END
  OBJ(battery)
  END
  OBJ(cpu)
    need_cpu_usage = 1;
  END
  OBJ(cpubar)
    need_cpu_usage = 1;
    obj->data.i = arg ? atoi(arg) : 4;
  END
  OBJ(color)
    obj->data.l = arg ? get_color(arg) : default_fg_color;
  END
  OBJ(downspeed)
    need_net_stats = 1;
    obj->data.net = get_net_stat(arg);
  END
  OBJ(downspeedf)
    need_net_stats = 1;
    obj->data.net = get_net_stat(arg);
  END
#ifdef HAVE_POPEN
  OBJ(exec)
    obj->data.s = strdup(arg ? arg : "");
  END
  OBJ(pre_exec)
    obj->type = OBJ_text;
    if (arg) {
      FILE *fp = popen(arg, "r");
      unsigned int n;
      char buf[2048];

      n = fread(buf, 1, 2048, fp);
      buf[n] = '\0';

      (void) pclose(fp);

      obj->data.s = strdup(buf);
    }
    else
      obj->data.s = strdup("");
  END
#endif
  OBJ(loadavg)
    int a = 1, b = 2, c = 3, r = 3;
    if (arg) {
      r = sscanf(arg, "%d %d %d", &a, &b, &c);
      a--; b--; c--;
      if (r >= 3 && (c < 0 || c >= 3))
        r--;
      if (r >= 2 && (b < 0 || b >= 3))
        r--, b = c;
      if (r >= 1 && (a < 0 || a >= 3))
        r--, a = b, b = c;
    }
    obj->data.loadavg[0] = (r >= 1) ? (unsigned char) a : 0;
    obj->data.loadavg[1] = (r >= 2) ? (unsigned char) b : 0;
    obj->data.loadavg[2] = (r >= 3) ? (unsigned char) c : 0;
  END
  OBJ(hr)
    obj->data.i = arg ? atoi(arg) : 1;
  END
  OBJ(i2c)
    char buf1[64], buf2[64];

    if(sscanf(arg, "%63s %63s %d", buf1, buf2, &obj->data.i2c.n) != 3) {
      /* if scanf couldn't read three values, read type and num and use
       * default device */
      sscanf(arg, "%63s %d", buf2, &obj->data.i2c.n);
      obj->data.i2c.dev = 0;
      obj->data.i2c.type = strdup(buf2);
    }
    else {
      obj->data.i2c.dev = strdup(buf1);
      obj->data.i2c.type = strdup(buf2);
    }
  END
  OBJ(kernel)
  END
  OBJ(machine)
  END
  OBJ(mails)
    need_mails = 1;
  END
  OBJ(mem)
    need_meminfo = 1;
  END
  OBJ(memmax)
    need_meminfo = 1;
  END
  OBJ(memperc)
    need_meminfo = 1;
  END
  OBJ(membar)
    need_meminfo = 1;
    obj->data.i = arg ? atoi(arg) : 4;
  END
  OBJ(new_mails) END
  OBJ(nodename) END
  OBJ(processes) END
  OBJ(running_processes) END
  OBJ(shadecolor)
    obj->data.l = arg ? get_color(arg) : default_bg_color;
  END
  OBJ(stippled_hr)
    int a = stippled_borders, b = 1;
    if(arg) {
      if(sscanf(arg, "%d %d", &a, &b) != 2)
        sscanf(arg, "%d", &b);
    }
    if (a <= 0) a = 1;
    obj->data.pair.a = a;
    obj->data.pair.b = b;
  END
  OBJ(swap) need_meminfo = 1; END
  OBJ(swapmax) need_meminfo = 1; END
  OBJ(swapperc) need_meminfo = 1; END
  OBJ(swapbar)
    need_meminfo = 1;
    obj->data.i = arg ? atoi(arg) : 4;
  END
  OBJ(sysname) END
  OBJ(temp1)
    obj->type = OBJ_i2c;
    obj->data.i2c.type = strdup("temp");
    obj->data.i2c.n = 1;
  END
  OBJ(temp2)
    obj->type = OBJ_i2c;
    obj->data.i2c.type = strdup("temp");
    obj->data.i2c.n = 2;
  END
  OBJ(time)
    obj->data.s = strdup(arg ? arg : "%F %T");
  END
  OBJ(totaldown)
    need_net_stats = 1;
    obj->data.net = get_net_stat(arg);
  END
  OBJ(totalup)
    need_net_stats = 1;
    obj->data.net = get_net_stat(arg);
  END
  OBJ(updates) END
  OBJ(upspeed)
    need_net_stats = 1;
    obj->data.net = get_net_stat(arg);
  END
  OBJ(upspeedf)
    need_net_stats = 1;
    obj->data.net = get_net_stat(arg);
  END
  OBJ(uptime)
  END
  {
    char buf[256];
    ERR("unknown variable %s", s);
    obj->type = OBJ_text;
    snprintf(buf, 256, "${%s}", s);
    obj->data.s = strdup(buf);
  }
#undef OBJ
}

/* append_text() appends text to last text_object if it's text, it it isn't
 * it creates a new text_object */
static void append_text(const char *s) {
  struct text_object *obj;

  if (s == NULL || *s == '\0')
    return;

  obj = text_object_count ? &text_objects[text_object_count-1] : 0;

  /* create a new text object? */
  if (!obj || obj->type != OBJ_text) {
    obj = text_object_new();
    obj->type = OBJ_text;
    obj->data.s = strdup(s);
  }
  else {
    /* append */
    obj->data.s = realloc(obj->data.s, strlen(obj->data.s) + strlen(s) + 1);
    strcat(obj->data.s, s);
  }
}

static void extract_variable_text(const char *p) {
  const char *s = p;

  free_text_objects();

  while (*p) {
    if (*p == '$') {
      * (char *) p = '\0';
      append_text(s);
      * (char *) p = '$';
      p++;
      s = p;

      if (*p != '$') {
        char buf[256];
        const char *var;
        unsigned int len;

        /* variable is either $foo or ${foo} */
        if (*p == '{') {
          p++;
          s = p;
          while (*p && *p != '}') p++;
        }
        else {
          s = p;
          if (*p == '#') p++;
          while (*p && (isalnum((int) *p) || *p=='_')) p++;
        }

        /* copy variable to buffer */
        len = (p - s > 255) ? 255 : (p - s);
        strncpy(buf, s, len);
        buf[len] = '\0';

        if (*p == '}') p++;
        s = p;

        var = getenv(buf);

        /* if variable wasn't found from environment, use some special */
        if (!var) {
          char *p;
          char *arg = 0;

          /* split arg */
          if (strchr(buf, ' ')) {
            arg = strchr(buf, ' ');
            *arg = '\0';
            arg++;
            while (isspace((int) *arg)) arg++;
            if (!*arg) arg = 0;
          }

          /* lowercase variable name */
          p = buf;
          while (*p) {
            *p = tolower(*p);
            p++;
          }

          construct_text_object(buf, arg);
        }
        continue;
      }
      else
        append_text("$");
    }

    p++;
  }
  append_text(s);
}

static void variable_substitute(const char *s, char *dest, unsigned int n) {
  while (*s && n > 1) {
    if (*s == '$') {
      s++;
      if (*s != '$') {
        char buf[256];
        const char *a, *var;
        unsigned int len;

        /* variable is either $foo or ${foo} */
        if (*s == '{') {
          s++;
          a = s;
          while (*s && *s != '}') s++;
        }
        else {
          a = s;
          if (*s == '#') s++;
          while (*s && (isalnum((int) *s) || *s=='_')) s++;
        }

        /* copy variable to buffer and look it up */
        len = (s-a > 255) ? 255 : (s-a);
        strncpy(buf, a, len);
        buf[len] = '\0';

        if (*s == '}') s++;

        var = getenv(buf);

        if (var) {
          /* add var to dest */
          len = strlen(var);
          if (len >= n) len = n-1;
          strncpy(dest, var, len);
          dest += len;
          n -= len;
        }
        continue;
      }
    }

    *dest++ = *s++;
    n--;
  }

  *dest = '\0';
}

double current_update_time, last_update_time;

static void generate_text() {
  unsigned int i, n;
  char *p;

  /* set net speeds to zero in case device was removed and doesn't get
   * updated */
  for (i=0; i<16; i++) {
    if (netstats[i].dev) {
      netstats[i].recv_speed = netstats[i].trans_speed = 0.0;
    }
  }

  special_count = 0;

  /* update info */

  current_update_time = get_time();

  prepare_update();

  if (need_cpu_usage) update_cpu_usage();
  if (need_net_stats) update_net_stats();
  if (need_mails) update_mail_count();
  if (need_meminfo) update_meminfo();

  /* generate text */

  n = TEXT_BUFFER_SIZE - 2;
  p = text_buffer;

  for (i=0; i<text_object_count; i++) {
    struct text_object *obj = &text_objects[i];

#define OBJ(a) break; case OBJ_##a:

    switch (obj->type) {
    default: {
      ERR("not implemented obj type %d", obj->type);
    }
    OBJ(acpitemp) {
      /* does anyone have decimals in acpi temperature? */
      snprintf(p, n, "%d", (int) get_acpi_temperature());
    }
    OBJ(battery) {
      get_battery_stuff(p, n);
    }
    OBJ(cpu) {
      snprintf(p, n, "%d", (int) (cpu_usage*100.0));
    }
    OBJ(cpubar) {
      new_bar(p, obj->data.i, (int) (cpu_usage*255.0));
    }
    OBJ(color) {
      new_fg(p, obj->data.l);
    }
    OBJ(downspeed) {
      snprintf(p, n, "%d", (int) (obj->data.net->recv_speed/1024));
    }
    OBJ(downspeedf) {
      snprintf(p, n, "%.1f", obj->data.net->recv_speed/1024.0);
    }
#ifdef HAVE_POPEN
    OBJ(exec) {
      FILE *fp = popen(obj->data.s, "r");
      p[fread(p, 1, n, fp)] = '\0';
      (void) pclose(fp);
      /* should change \001 to spaces here */
    }
#endif
    OBJ(loadavg) {
      double v[3];
      get_load_average(v);

      if (obj->data.loadavg[2])
        snprintf(p, n, "%.2f %.2f %.2f", v[obj->data.loadavg[0]-1],
            v[obj->data.loadavg[1]-1], v[obj->data.loadavg[2]-1]);
      else if (obj->data.loadavg[1])
        snprintf(p, n, "%.2f %.2f", v[obj->data.loadavg[0]-1],
            v[obj->data.loadavg[1]-1]);
      else if (obj->data.loadavg[0])
        snprintf(p, n, "%.2f", v[obj->data.loadavg[0]-1]);
    }
    OBJ(hr) {
      new_hr(p, obj->data.i);
    }
    OBJ(i2c) {
      double r;

      r = get_i2c_info(&obj->data.i2c.fd, obj->data.i2c.dev,
          obj->data.i2c.type, obj->data.i2c.n);

      /* put integer if it's fan (RPM), else float (temp or vol) */
      if (strcmp(obj->data.i2c.type, "fan") == 0)
        snprintf(p, n, "%d", (int) r);
      else
        snprintf(p, n, "%.1f", r);
    }
    OBJ(kernel) {
      snprintf(p, n, "%s", uname_s.release);
    }
    OBJ(machine) {
      snprintf(p, n, "%s", uname_s.machine);
    }
    OBJ(mails) {
      snprintf(p, n, "%d", mail_count);
    }
    OBJ(mem) {
      human_readable(mem*1024, p);
    }
    OBJ(memmax) {
      human_readable(memmax*1024, p);
    }
    OBJ(memperc) {
      if (memmax)
        snprintf(p, n, "%d", (mem*100) / (memmax));
    }
    OBJ(membar) {
      new_bar(p, obj->data.i, memmax ? (mem*255) / (memmax) : 0);
    }
    OBJ(new_mails) {
      snprintf(p, n, "%d", new_mail_count);
    }
    OBJ(nodename) {
      snprintf(p, n, "%s", uname_s.nodename);
    }
    OBJ(processes) {
      snprintf(p, n, "%d", get_total_processes());
    }
    OBJ(running_processes) {
      snprintf(p, n, "%d", get_running_processes());
    }
    OBJ(text) {
      snprintf(p, n, "%s", obj->data.s);
    }
    OBJ(shadecolor) {
      new_bg(p, obj->data.l);
    }
    OBJ(stippled_hr) {
      new_stippled_hr(p, obj->data.pair.a, obj->data.pair.b);
    }
    OBJ(swap) {
      human_readable(swap*1024, p);
    }
    OBJ(swapmax) {
      human_readable(swapmax*1024, p);
    }
    OBJ(swapperc) {
      if (swapmax)
        snprintf(p, 255, "%u", (swap*100) / (swapmax));
    }
    OBJ(swapbar) {
      new_bar(p, obj->data.i, swapmax ? (swap*255) / (swapmax) : 0);
    }
    OBJ(sysname) {
      snprintf(p, n, "%s", uname_s.sysname);
    }
    OBJ(time) {
      time_t t = time(NULL);
      struct tm *tm = localtime(&t);
      strftime(p, n, obj->data.s, tm);
    }
    OBJ(totaldown) {
      human_readable(obj->data.net->recv, p);
    }
    OBJ(totalup) {
      human_readable(obj->data.net->trans, p);
    }
    OBJ(updates) {
      static int total_updates = 0;
      snprintf(p, n, "%d", total_updates++);
    }
    OBJ(upspeed) {
      snprintf(p, n, "%d", (int) (obj->data.net->trans_speed/1024));
    }
    OBJ(upspeedf) {
      snprintf(p, n, "%.1f", obj->data.net->trans_speed/1024.0);
    }
    OBJ(uptime) {
      double t = get_uptime();
      format_time(p, n, (int) t);
    }
    break;
    }

    {
      unsigned int a = strlen(p);
      p += a;
      n -= a;
    }
  }

  last_update_time = current_update_time;

  if (stuff_in_upper_case) {
    char *p;

    p = text_buffer;
    while (*p) {
      *p = toupper(*p);
      p++;
    }
  }
}

/*
 * text size
 */

#define BORDER 3

static int text_start_x, text_start_y; /* text start position in window */
static int text_width, text_height;

static inline int get_string_width(const char *s) {
  return *s ? XTextWidth(font_info, s, strlen(s)) : 0;
}

static void text_size_updater(char *s) {
  int w = 0;
  char *p;

  /* get string widths and skip specials */
  p = s;
  while (*p) {
    if (*p == SPECIAL_CHAR) {
      *p = '\0';
      w += get_string_width(s);
      *p = SPECIAL_CHAR;
      s = p+1;
    }
    p++;
  }

  w += get_string_width(s);

  if (w > text_width) text_width = w;

  text_height += font_info->max_bounds.ascent +
                 font_info->max_bounds.descent;
}

static void update_text_area() {
  int x, y;

  /* update text size if it isn't fixed */
  if (!fixed_size) {
    text_width = 1;
    text_height = 1;
    for_each_line(text_buffer, text_size_updater);
  }

  /* get text position on workarea */
  switch (text_alignment) {
  case TOP_LEFT:
    x = gap_x;
    y = gap_y;
    break;

  case TOP_RIGHT:
    x = workarea[2] - text_width - gap_x;
    y = gap_y;
    break;

  default:
  case BOTTOM_LEFT:
    x = gap_x;
    y = workarea[3] - text_height - gap_y;
    break;

  case BOTTOM_RIGHT:
    x = workarea[2] - text_width - gap_x;
    y = workarea[3] - text_height - gap_y;
    break;
  }

#ifdef OWN_WINDOW
  if (own_window) {
    x += workarea[0];
    y += workarea[1];
    text_start_x = BORDER;
    text_start_y = BORDER;
    window_x = x - BORDER;
    window_y = y - BORDER;
  }
  else
#endif
  {
    /* If window size doesn't match to workarea's size, then window
     * probably includes panels (gnome).
     * Blah, doesn't work on KDE. */
#if 0
    if (workarea[2] != window_width) x += workarea[0];
    if (workarea[3] != window_height) y += workarea[1];
#endif

    text_start_x = x;
    text_start_y = y;
  }
}

/*
 * drawing stuff
 */

static int cur_x, cur_y; /* current x and y for drawing */
static int specials_used;
static int draw_mode; /* FG or BG */

static inline void set_foreground_color(unsigned long c) {
  XSetForeground(display, gc, c);
}

static void draw_string(const char *s) {
  if (s[0] == '\0') return;

  XDrawString(display, win, gc, cur_x, cur_y, s, strlen(s));

  cur_x += get_string_width(s);
}

static void draw_line(char *s) {
  char *p;

  cur_x = text_start_x;
  cur_y += font_info->max_bounds.ascent;

  /* find specials and draw stuff */
  p = s;
  while (*p) {
    if (*p == SPECIAL_CHAR) {
      int w = 0;

      /* draw string before special */
      *p = '\0';
      draw_string(s);
      *p = SPECIAL_CHAR;
      s = p+1;

      /* draw special */
      switch (specials[specials_used].type) {
      case HORIZONTAL_LINE:
        {
          int h = specials[specials_used].height;
          int mid = font_info->max_bounds.ascent / 2;
          w = text_start_x + text_width - cur_x;

          XSetLineAttributes(display, gc, h, LineSolid, CapButt, JoinMiter);
          XDrawLine(display, win, gc, cur_x, cur_y-mid, cur_x+w, cur_y-mid);
        }
        break;

      case STIPPLED_HR:
        {
          int h = specials[specials_used].height;
          int s = specials[specials_used].arg;
          int mid = font_info->max_bounds.ascent / 2;
          w = text_start_x + text_width - cur_x - 1;

          XSetLineAttributes(display, gc, h, LineOnOffDash, CapButt, JoinMiter);
          XSetDashes(display, gc, 0, (char[]){s, s}, 2);
          XDrawLine(display, win, gc, cur_x, cur_y-mid, cur_x+w, cur_y-mid);
        }
        break;

      case BAR:
        {
          int h = specials[specials_used].height;
          int bar_usage = specials[specials_used].arg;
          int by = cur_y - (font_info->max_bounds.ascent + h)/2 + 1;
          w = text_start_x + text_width - cur_x - 1;

          XSetLineAttributes(display, gc, 1, LineSolid, CapButt, JoinMiter);

          XDrawRectangle(display, win, gc, cur_x, by, w, h);
          XFillRectangle(display, win, gc, cur_x, by, w * bar_usage / 255, h);
        }
        break;

      case FG:
        if (draw_mode == FG)
          set_foreground_color(specials[specials_used].arg);
        break;

      case BG:
        if (draw_mode == BG)
          set_foreground_color(specials[specials_used].arg);
        break;
      }

      cur_x += w;

      specials_used++;
    }

    p++;
  }

  draw_string(s);

  cur_y += font_info->max_bounds.descent;
}

static void draw_text() {
  cur_y = text_start_y;

  /* draw borders */
  if (draw_borders) {
    if(stippled_borders) {
      XSetLineAttributes(display, gc, 1, LineOnOffDash, CapButt, JoinMiter);
      XSetDashes(display, gc, 0, (char[]){stippled_borders, stippled_borders}, 2);
    }
    else {
      XSetLineAttributes(display, gc, 1, LineSolid, CapButt, JoinMiter);
    }

    XDrawRectangle(display, win, gc, text_start_x-2, text_start_y-2,
        text_width+3, text_height+3);
  }

  /* draw text */
  specials_used = 0;
  for_each_line(text_buffer, draw_line);
}

static void draw_stuff() {
  XSetLineAttributes(display, gc, 1, LineSolid, CapButt, JoinMiter);

  text_start_x++;
  text_start_y++;
  if (draw_shades) {
    set_foreground_color(default_bg_color);
    draw_mode = BG;
    draw_text();
  }

  text_start_x--;
  text_start_y--;
  set_foreground_color(default_fg_color);
  draw_mode = FG;
  draw_text();
}

static void clear_text() {
  XClearArea(display, win, text_start_x-BORDER, text_start_y-BORDER,
      text_width+BORDER*2, text_height+BORDER*2, True);
}

static int need_to_update;

/* update_text() generates new text and clears old text area */
static void update_text() {
  generate_text();
  clear_text();
  need_to_update = 1;
}

static void load_config_file(const char *);

/* handles SIGHUPs by reloading config */
static void hup_handler(int a) {
  if (current_config) {
    load_config_file(current_config);
    extract_variable_text(text);
    free(text);
    text = NULL;
    update_text();
  }
}

static void clean_up() {
#ifdef OWN_WINDOW
  if (own_window)
    XDestroyWindow(display, win);
  else
#endif
  {
    clear_text();
    XFlush(display);
  }

  XFreeGC(display, gc);

  /* it is really pointless to free() memory at the end of program but ak|ra
   * wants me to do this */

  free_text_objects();

  if (text != original_text)
    free(text);

  free(current_config);
  free(current_mail_spool);
}

static void term_handler(int a) {
  clean_up();
  exit(0);
}

static void main_loop() {
  Region region = XCreateRegion();

  while (1) {
    XFlush(display);

    /* wait for X event or timeout */

    if (!XPending(display)) {
      fd_set fdsr;
      struct timeval tv;
      int s;
      double t = update_interval - (get_time() - last_update_time);

      if (t < 0) t = 0;

      tv.tv_sec = (long) t;
      tv.tv_usec = (long) (t * 1000000) % 1000000;

      FD_ZERO(&fdsr);
      FD_SET(ConnectionNumber(display), &fdsr);

      s = select(ConnectionNumber(display) + 1, &fdsr, 0, 0, &tv);
      if (s == -1) {
        if (errno != EINTR)
          ERR("can't select(): %s", strerror(errno));
      }
      else {
        /* timeout */
        if (s == 0)
          update_text();
      }
    }

    /* handle X events */

    while (XPending(display)) {
      XEvent ev;
      XNextEvent(display, &ev);

      switch (ev.type) {
      case Expose: {
          XRectangle r;
          r.x = ev.xexpose.x;
          r.y = ev.xexpose.y;
          r.width = ev.xexpose.width;
          r.height = ev.xexpose.height;
          XUnionRectWithRegion(&r, region, region);
        }
        break;

#ifdef OWN_WINDOW
      case ReparentNotify:
        /* set background to ParentRelative for all parents */
        if (own_window) {
          unsigned int i;
          Window parent = win;

          for (i=0; i<10; i++) {
            Window r, *children;
            unsigned int n;

            XQueryTree(display, parent, &r, &parent, &children, &n);
            XFree(children);

            if (parent == RootWindow(display, screen)) break;
            XSetWindowBackgroundPixmap(display, parent, ParentRelative);
          }

          XClearWindow(display, win);
        }
        break;

      case ConfigureNotify:
        if (own_window) {
          if (ev.xconfigure.width != window_width ||
              ev.xconfigure.height != window_height) {
            if (window_width != 0 && window_height != 0)
              fixed_size = 1;

            clear_text();

            {
              XWindowAttributes attrs;
              if (XGetWindowAttributes(display, win, &attrs)) {
                window_width = attrs.width;
                window_height = attrs.height;
              }
            }

            text_width = window_width - BORDER*2;
            text_height = window_height - BORDER*2;
          }
        }
        break;
#endif

      default:
        break;
      }
    }

    if (need_to_update) {
      need_to_update = 0;

      update_text_area();

#ifdef OWN_WINDOW
      if (own_window) {
        if (!fixed_size &&
            (text_width+BORDER*2 != window_width || text_height+BORDER*2 != window_height)) {
          window_width = text_width + BORDER*2;
          window_height = text_height + BORDER*2;
          XResizeWindow(display, win, window_width, window_height);
        }
      }
#endif

      XRectangle r;
      r.x = text_start_x-BORDER;
      r.y = text_start_y-BORDER;
      r.width = text_width+BORDER*2;
      r.height = text_height+BORDER*2;
      XUnionRectWithRegion(&r, region, region);
    }

    if (!XEmptyRegion(region)) {
      XSetRegion(display, gc, region);
      draw_stuff();
      XDestroyRegion(region);
      region = XCreateRegion();
      XFlush(display);
    }
  }
}

#define ATOM(a) XInternAtom(display, #a, False)

static Window find_subwindow(Window win, int w, int h) {
  unsigned int i, j;
  Window troot, parent, *children;
  unsigned int n;

  /* search subwindows with same size as display or work area */

  for (i=0; i<10; i++) {
    XQueryTree(display, win, &troot, &parent, &children, &n);

    for (j=0; j<n; j++) {
      XWindowAttributes attrs;

      if (XGetWindowAttributes(display, children[j], &attrs)) {
        if ((attrs.width == display_width && attrs.height == display_height) ||
            (attrs.width == w && attrs.height == h)) {
          win = children[j];
          break;
        }
      }
    }

    XFree(children);
    if (j == n) break;
  }

  return win;
}

static void update_workarea() {
  Window root = RootWindow(display, screen);
  unsigned long nitems, bytes;
  unsigned char *buf = NULL;
  Atom type;
  int format;

  /* default work area is display */
  workarea[0] = 0;
  workarea[1] = 0;
  workarea[2] = display_width;
  workarea[3] = display_height;

  /* get current desktop */
  if (XGetWindowProperty(display, root, ATOM(_NET_CURRENT_DESKTOP),
        0, 1, False, XA_CARDINAL, &type, &format, &nitems, &bytes, &buf) ==
      Success && type == XA_CARDINAL && nitems > 0) {
    long desktop = * (long *) buf;

    XFree(buf); buf = 0;

    /* get workarea */
    if (XGetWindowProperty(display, root, ATOM(_NET_WORKAREA), desktop*4, 4,
          False, XA_CARDINAL, &type, &format, &nitems, &bytes, &buf) ==
        Success && type == XA_CARDINAL) {
      workarea[0] = ((long *) buf)[0];
      workarea[1] = ((long *) buf)[1];
      workarea[2] = ((long *) buf)[2];
      workarea[3] = ((long *) buf)[3];
    }
  }

  if (buf) { XFree(buf); buf = 0; }
}

static Window find_window_to_draw() {
  Atom type;
  int format, i;
  unsigned long nitems, bytes;
  unsigned int n;
  Window root = RootWindow(display, screen);
  Window win = root;
  Window troot, parent, *children;
  unsigned char *buf = NULL;

  /* some window managers set __SWM_VROOT to some child of root window */

  XQueryTree(display, root, &troot, &parent, &children, &n);
  for (i=0; i<(int)n; i++) {
    if (XGetWindowProperty(display, children[i], ATOM(__SWM_VROOT),
          0, 1, False, XA_WINDOW, &type, &format, &nitems, &bytes,
          &buf) == Success && type == XA_WINDOW) {
      win = * (Window *) buf;
      XFree(buf);
      XFree(children);
      fprintf(stderr, "torsmo: drawing to window from __SWM_VROOT property\n");
      return win;
    }

    if (buf) {
      XFree(buf);
      buf = 0;
    }
  }
  XFree(children);

  /* get subwindows from root */
  win = find_subwindow(root, -1, -1);

  update_workarea();

  win = find_subwindow(win, workarea[2], workarea[3]);

  if (buf) { XFree(buf); buf = 0; }

  if (win != root)
    fprintf(stderr, "torsmo: drawing to subwindow of root window\n");
  else
    fprintf(stderr, "torsmo: drawing to root window\n");

  return win;
}

static void init_X() {
  if ((display=XOpenDisplay(0)) == NULL)
    CRIT_ERR("can't open display: %s", XDisplayName(0));

  screen = DefaultScreen(display);
  display_width = DisplayWidth(display, screen);
  display_height = DisplayHeight(display, screen);

  default_bg_color = BlackPixel(display, screen);
  default_fg_color = WhitePixel(display, screen);

  update_workarea();
}

static void init_window() {
#ifdef OWN_WINDOW
  if (own_window) {
    /* look like root pixmap isn't needed for anything */
#if 0
    /* get root pixmap */
    {
      Atom type;
      int format;
      unsigned long nitems, bytes;
      unsigned char *buf = NULL;

      if (XGetWindowProperty (display, RootWindow(display, screen),
            ATOM(_XROOTPMAP_ID), 0, 1, False, XA_PIXMAP, &type, &format,
            &nitems, &bytes, &buf) == Success) {
        if (buf) {
          root_pixmap = * (Pixmap *) buf;
          XFree(buf);
        }
      }
    }
#endif

    {
      XSetWindowAttributes attrs;

      attrs.background_pixel = get_color("green");

      win = XCreateWindow(display, RootWindow(display, screen),
          window_x, window_y, text_width+8, text_height+8, 0,
          CopyFromParent, /* depth */
          CopyFromParent, /* class */
          CopyFromParent, /* visual */
          (0*CWOverrideRedirect) | (CWBackPixel),
          &attrs);

      XStoreName(display, win, "Torsmo");

      XClearWindow(display, win);

      XMoveWindow(display, win, window_x, window_y);
    }

    XSetWindowBackgroundPixmap(display, win, ParentRelative);

    {
      /* turn off decorations */
      Atom a = XInternAtom(display, "_MOTIF_WM_HINTS", True);
      if (a != None) {
        long prop[5] = { 2, 0, 0, 0, 0 };
        XChangeProperty(display, win, a, a, 32, PropModeReplace,
            (unsigned char *) prop, 5);
      }
    }
  }
  else
#endif
  {
    XWindowAttributes attrs;

    if (!win)
      win = find_window_to_draw();

    if (XGetWindowAttributes(display, win, &attrs)) {
      window_width = attrs.width;
      window_height = attrs.height;
    }
  }

  XSelectInput(display, win, ExposureMask
#ifdef OWN_WINDOW
      | (own_window ? (StructureNotifyMask | PropertyChangeMask) : 0)
#endif
      );

  XMapWindow(display, win);
}

static int string_to_bool(const char *s) {
  if (!s) return 1;
  if (strcasecmp(s, "yes") == 0) return 1;
  if (strcasecmp(s, "true") == 0) return 1;
  if (strcasecmp(s, "1") == 0) return 1;
  return 0;
}

static void load_config_file(const char *f) {
  int line = 0;
  FILE *fp = open_file(f, 0);
  if (!fp) return;

  while (!feof(fp)) {
    char buf[256], *p, *p2, *name, *value;
    line++;
    if (fgets(buf, 256, fp) == NULL) break;

    p = buf;

    /* break at comment */
    p2 = strchr(p, '#');
    if (p2) *p2 = '\0';

    /* skip spaces */
    while (*p && isspace((int) *p)) p++;
    if (*p == '\0') continue; /* empty line */

    name = p;

    /* skip name */
    p2 = p;
    while (*p2 && !isspace((int) *p2)) p2++;
    if (*p2 != '\0') {
      *p2 = '\0'; /* break at name's end */
      p2++;
    }

    /* get value */
    if (*p2) {
      p = p2;
      while (*p && isspace((int) *p)) p++;

      value = p;

      p2 = value + strlen(value);
      while (isspace((int) *(p2-1))) *--p2 = '\0';
    }
    else {
      value = 0;
    }

#define WTF ERR("%s: %d: wtf?", f, line);

    if (strcasecmp(name, "alignment") == 0) {
      if (value) {
        if (strcasecmp(value, "top_left") == 0) text_alignment = TOP_LEFT;
        else if (strcasecmp(value, "top_right") == 0) text_alignment = TOP_RIGHT;
        else if (strcasecmp(value, "bottom_left") == 0) text_alignment = BOTTOM_LEFT;
        else if (strcasecmp(value, "bottom_right") == 0) text_alignment = BOTTOM_RIGHT;
        else WTF
      }
      else
        WTF
    }
    else if (strcasecmp(name, "background") == 0) {
      fork_to_background = string_to_bool(value);
    }
    else if (strcasecmp(name, "default_color") == 0) {
      if (value)
        default_fg_color = get_color(value);
      else
        WTF
    }
    else if (strcasecmp(name, "default_shadecolor") == 0) {
      if (value)
        default_bg_color = get_color(value);
      else
        WTF
    }
    else if (strcasecmp(name, "draw_borders") == 0) {
      draw_borders = string_to_bool(value);
    }
    else if (strcasecmp(name, "draw_shades") == 0) {
      draw_shades = string_to_bool(value);
    }
    else if (strcasecmp(name, "font") == 0) {
      if (value)
        font = strdup(value);
      else
        WTF
    }
    else if (strcasecmp(name, "gap_x") == 0) {
      if (value)
        gap_x = atoi(value);
      else
        WTF
    }
    else if (strcasecmp(name, "gap_y") == 0) {
      if (value)
        gap_y = atoi(value);
      else
        WTF
    }
    else if (strcasecmp(name, "mail_spool") == 0) {
      if (value) {
        char buf[256];
        variable_substitute(value, buf, 256);

        if (buf[0] != '\0') {
          if (current_mail_spool) free(current_mail_spool);
          current_mail_spool = strdup(buf);
        }
      }
      else
        WTF
    }
    else if (strcasecmp(name, "no_buffers") == 0) {
      no_buffers = string_to_bool(value);
    }
#ifdef OWN_WINDOW
    else if (strcasecmp(name, "own_window") == 0) {
      own_window = string_to_bool(value);
    }
#endif
    else if (strcasecmp(name, "stippled_borders") == 0) {
      if(value)
        stippled_borders = strtol(value, 0, 0);
      else
        stippled_borders = 4;
    }
    else if (strcasecmp(name, "temp1") == 0) {
      ERR("temp1 configuration is obsolete, use ${i2c <i2c device here> temp 1}");
    }
    else if (strcasecmp(name, "temp2") == 0) {
      ERR("temp2 configuration is obsolete, use ${i2c <i2c device here> temp 2}");
    }
    else if (strcasecmp(name, "update_interval") == 0) {
      if (value)
        update_interval = strtod(value, 0);
      else
        WTF
    }
    else if (strcasecmp(name, "uppercase") == 0) {
      stuff_in_upper_case = string_to_bool(value);
    }
    else if (strcasecmp(name, "text") == 0) {
      if (text != original_text)
        free(text);

      text = (char *) malloc(1);
      text[0] = '\0';

      while (!feof(fp)) {
        unsigned int l = strlen(text);
        if (fgets(buf, 256, fp) == NULL) break;
        text = (char *) realloc(text, l + strlen(buf) + 1);
        strcat(text, buf);

        if (strlen(text) > 1024*8) break;
      }
      fclose(fp);
      return;
    }
    else
      WTF
  }

  fclose(fp);
}

static const char *getopt_string = "vVdt:f:u:hc:w:"
#ifdef OWN_WINDOW
"o"
#endif
;

int main(int argc, char **argv) {
  /* handle command line parameters that don't change configs */

  while (1) {
    int c = getopt(argc, argv, getopt_string);
    if (c == -1) break;

    switch (c) {
    case 'v':
    case 'V':
      printf("torsmo " VERSION " compiled " __DATE__ "\n");
      return 0;

    case 'c':
      /* if current_config is set to a strdup of CONFIG_FILE, free it (even
       * though free() does the NULL check itself;), then load optarg value */
      if (current_config) free(current_config);
      current_config = strdup(optarg);
      break;

    case 'h':
      printf(
"Usage: %s [OPTION]...\n"
"   -V            version\n"
"   -c FILE       config file to load instead of " CONFIG_FILE "\n"
"   -d            daemonize, fork to background\n"
"   -f FONT       font to use\n"
"   -h            help\n"
#ifdef OWN_WINDOW
"   -o            create own window to draw\n"
#endif
"   -t TEXT       text to render\n"
"   -u SECS       update interval\n"
"   -w WIN_ID     window id to draw\n"
, argv[0]);
      return 0;

    case 'w':
      win = strtol(optarg, 0, 0);
      break;

    case '?':
      exit(EXIT_FAILURE);
    }
  }

  /* X stuff must be initialized here to get default colors */
  init_X();

  /* load current_config or CONFIG_FILE */

#ifdef CONFIG_FILE
  if (current_config == NULL)
  {
    /* load default config file */
    char buf[256];

    variable_substitute(CONFIG_FILE, buf, 256);

    if (buf[0] != '\0')
      current_config = strdup(buf);
  }
#endif

  if (current_config != NULL)
    load_config_file(current_config);

#ifdef MAIL_FILE
  if (current_mail_spool == NULL) {
    char buf[256];
    variable_substitute(MAIL_FILE, buf, 256);

    if (buf[0] != '\0')
      current_mail_spool = strdup(buf);
  }
#endif

  /* handle other command line arguments */

  optind = 0;

  while (1) {
    int c = getopt(argc, argv, getopt_string);
    if(c == -1) break;

    switch (c) {
    case 'd':
      fork_to_background = 1;
      break;

    case 'f':
      font = optarg;
      break;

#ifdef OWN_WINDOW
    case 'o':
      own_window = 1;
      break;
#endif

    case 't':
      if (text != original_text) free(text);
      text = strdup(optarg);
      convert_escapes(text);
      break;

    case 'u':
      update_interval = strtod(optarg, 0);
      break;

    case '?':
      exit(EXIT_FAILURE);
    }
  }

  /* load font */
  if ((font_info = XLoadQueryFont(display, font)) == NULL) {
    ERR("can't load font '%s'", font);
    if ((font_info = XLoadQueryFont(display, "fixed")) == NULL) {
      CRIT_ERR("can't load font '%s'", "fixed");
    }
  }

  /* generate text and get initial size */
  extract_variable_text(text);
  if (text != original_text) free(text);
  text = NULL;

  update_uname();
  generate_text();
  update_text_area();

/*#define BENCHMARK*/
#ifdef BENCHMARK
  {
    unsigned int i;
    for (i=0; i<100000; i++) {
      generate_text();
    }
    fprintf(stderr, "%s\n", text_buffer);
  }
  return 0;
#endif

  init_window();

  /* prepare GC */
  {
    XGCValues values;
    values.font = font_info->fid;
    values.graphics_exposures = 0;
    values.function = GXcopy;
    gc = XCreateGC(display, win, GCFunction | GCFont | GCGraphicsExposures, &values);
  }

  draw_stuff();

  /* fork */
  if (fork_to_background) {
    int ret = fork();
    switch (ret) {
    case -1:
      ERR("can't fork() to background: %s", strerror(errno));
      break;

    case 0:
      break;

    default:
      fprintf(stderr, "torsmo: forked to background, pid is %d\n", ret);
      exit(0);
      break;
    }
  }

  /* set SIGHUP, SIGINT and SIGTERM handler */
  {
    struct sigaction sa;

    sa.sa_handler = hup_handler;
    sigemptyset(&sa.sa_mask);
    sa.sa_flags = SA_RESTART;
    if (sigaction(SIGHUP, &sa, NULL) != 0) {
      ERR("can't set signal handler for SIGHUP: %s", strerror(errno));
    }

    sa.sa_handler = term_handler;
    sigemptyset(&sa.sa_mask);
    sa.sa_flags = SA_RESTART;

    if (sigaction(SIGINT, &sa, NULL) != 0) {
      ERR("can't set signal handler for SIGINT: %s", strerror(errno));
    }

    sa.sa_handler = term_handler;
    sigemptyset(&sa.sa_mask);
    sa.sa_flags = SA_RESTART;

    if (sigaction(SIGTERM, &sa, NULL) != 0) {
      ERR("can't set signal handler for SIGTERM: %s", strerror(errno));
    }
  }

  main_loop();

  return 0;
}
