#include "torsmo.h"
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <dirent.h>
#include <ctype.h>
#include <errno.h>
#include <limits.h>
#include <sys/types.h>
#include <sys/sysinfo.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>

static int sysinfo_updated;
static struct sysinfo s_info;

void prepare_update() {
  sysinfo_updated = 0;
}

static void update_sysinfo() {
  if (sysinfo_updated) return;
  sysinfo_updated = 1;

  sysinfo(&s_info);
}

double get_uptime() {
  double t = 0;

  /* prefers sysinfo() for uptime, I don't really know which one is better
   * (=faster?) */

#ifdef USE_PROC_UPTIME
  static int rep;
  FILE *fp = open_file("/proc/uptime", &rep);
  if(!fp) return 0;
  fscanf(fp, "%lf", &t);
  fclose(fp);
#else
  update_sysinfo();
  t = (double) s_info.uptime;
#endif

  return t;
}

/* these things are also in sysinfo except Buffers:, that's why I'm reading
 * them from proc */

static FILE *meminfo_fp;

void update_meminfo() {
  static int rep;
  unsigned int a;
  char buf[256];

  mem = memmax = swap = swapmax = bufmem = 0;

  if (meminfo_fp == NULL)
    meminfo_fp = open_file("/proc/meminfo", &rep);
  else
    fseek(meminfo_fp, 0, SEEK_SET);
  if (meminfo_fp == NULL) return;

  while (!feof(meminfo_fp)) {
    if (fgets(buf, 255, meminfo_fp) == NULL) break;

    if (strncmp(buf, "MemTotal:", 9) == 0) {
      sscanf(buf, "%*s %u", &memmax);
    }
    else if (strncmp(buf, "MemFree:", 8) == 0) {
      sscanf(buf, "%*s %u", &mem);
    }
    else if (strncmp(buf, "SwapTotal:", 10) == 0) {
      sscanf(buf, "%*s %u", &swapmax);
    }
    else if (strncmp(buf, "SwapFree:", 9) == 0) {
      sscanf(buf, "%*s %u", &swap);
    }
    else if (strncmp(buf, "Buffers:", 8) == 0) {
      sscanf(buf, "%*s %u", &a);
      bufmem += a;
    }
    else if (strncmp(buf, "Cached:", 7) == 0) {
      sscanf(buf, "%*s %u", &a);
      bufmem += a;
    }
  }

  /* fclose(fp); meminfo_fp = NULL; */

  mem = memmax - mem;
  swap = swapmax - swap;

  if (no_buffers) mem -= bufmem;
}

static FILE *net_dev_fp;

void update_net_stats() {
  static int rep;
  unsigned int i;
  char buf[256];
  double delta;

  /* get delta */
  delta = current_update_time - last_update_time;
  if (delta <= 0.0001) return;

  /* open file and ignore first two lines */
  if (net_dev_fp == NULL)
    net_dev_fp = open_file("/proc/net/dev", &rep);
  else
    fseek(net_dev_fp, 0, SEEK_SET);
  if (!net_dev_fp) return;

  fgets(buf, 255, net_dev_fp); /* garbage */
  fgets(buf, 255, net_dev_fp); /* garbage (field names) */

  /* read each interface */
  for (i=0; i<16; i++) {
    struct net_stat *ns;
    char *s, *p;
    long long r, t, last_recv, last_trans;

    if (fgets(buf, 255, net_dev_fp) == NULL) break;
    p = buf;
    while (isspace((int) *p)) p++;

    s = p;

    while (*p && *p != ':') p++;
    if (*p == '\0') continue;
    *p = '\0';
    p++;

    ns = get_net_stat(s);
    last_recv = ns->recv;
    last_trans = ns->trans;

    sscanf(p,
    /* bytes packets errs drop fifo frame compressed multicast|bytes ... */
       "%Ld  %*d     %*d  %*d  %*d  %*d   %*d        %*d       %Ld",
        &r,                                                    &t);

    /* if recv or trans is less than last time, an overflow happened */

    if (r < ns->last_read_recv)
      ns->recv += ((long long) 4294967295U - ns->last_read_recv) + r;
    else
      ns->recv += (r - ns->last_read_recv);
    ns->last_read_recv = r;

    if (t < ns->last_read_trans)
      ns->trans += ((long long) 4294967295U - ns->last_read_trans) + t;
    else
      ns->trans += (t - ns->last_read_trans);
    ns->last_read_trans = t;

    /* calculate speeds */
    ns->recv_speed = (ns->recv - last_recv) / delta;
    ns->trans_speed = (ns->trans - last_trans) / delta;
  }

  /* fclose(net_dev_fp); net_dev_fp = NULL; */
}

int get_total_processes() {
  update_sysinfo();
  return s_info.procs;
}

static int running_processes;
static unsigned int cpu_user, cpu_system, cpu_nice;

static FILE *stat_fp;

static void update_stat() {
  static int rep;
  char buf[256];

  if (stat_fp == NULL)
    stat_fp = open_file("/proc/stat", &rep);
  else
    fseek(stat_fp, 0, SEEK_SET);
  if (stat_fp == NULL) return;

  while (!feof(stat_fp)) {
    if (fgets(buf, 255, stat_fp) == NULL)
      break;

    if (strncmp(buf, "procs_running ", 14) == 0) {
      sscanf(buf, "%*s %d", &running_processes);
    }
    else if (strncmp(buf, "cpu ", 4) == 0) {
      sscanf(buf, "%*s %u %u %u", &cpu_user, &cpu_nice, &cpu_system);
    }
  }

  /* fclose(fp); */
}

int get_running_processes() {
  update_stat();
  return running_processes;
}

static double last_cpu_sum;

void update_cpu_usage() {
  double delta;

  delta = current_update_time - last_update_time;
  if(delta <= 0.001) return;

  update_stat();

  cpu_usage = (cpu_user+cpu_nice+cpu_system - last_cpu_sum) / delta / 100.0;
  last_cpu_sum = cpu_user+cpu_nice+cpu_system;
}

static const char *default_i2c_dev;

static int no_dots(const struct dirent *d) {
  if(d->d_name[0] == '.') return 0;
  return 1;
}

#define I2C_DIR "/sys/bus/i2c/devices/"

double get_i2c_info(int *fd, const char *dev, const char *type, int n) {
  char path[256];
  int val = 0;

  /* if i2c device is NULL or *, get first */
  if (dev == NULL || strcmp(dev, "*") == 0) {
    if (default_i2c_dev == NULL) {
      struct dirent **namelist;
      int i, n;

      n = scandir(I2C_DIR, &namelist, no_dots, alphasort);
      if (n < 0) {
        static int rep;
        if (!rep) {
          ERR("scandir for %s: %s", I2C_DIR, strerror(errno));
          rep = 1;
        }
        return 0;
      }
      else {
        if (n == 0) {
          /* no sensors, oh dear. */
          return 0;
        }

        /* get default i2c device and free others (leaks a bit but this is
         * done only once per run) */
        default_i2c_dev = namelist[0]->d_name;

        for (i=1; i<n; i++)
          free(namelist[i]);
        free(namelist);
      }
    }

    dev = default_i2c_dev;
  }

  /* change vol to in */
  if (strcmp(type, "vol") == 0)
    type = "in";

  snprintf(path, 255, "/sys/bus/i2c/devices/%s/%s%d_input", dev, type, n);

  /* open file */
  if (*fd == 0)
    *fd = open(path, O_RDONLY);
  else
    lseek(*fd, 0, SEEK_SET);
  if (!*fd) return 0;

  /* read integer */
  {
    char buf[64];
    unsigned int n;
    n = read(*fd, buf, 63);
    /* should read until n == 0 but I doubt that kernel will give these
     * in multiple pieces. :) */
    buf[n] = '\0';
    val = atoi(buf);
  }

  /* close(*fd); *fd = 0; */

  /* divide voltage and temperature by 1000 */
  if (strcmp(type, "in") == 0)
    return val / 1000.0;
  if (strcmp(type, "temp") == 0)
    return val / 1000.0;
  else
    return val;
}

void get_load_average(double v[3]) {
#ifdef HAVE_GETLOADAVG
  getloadavg(v, 3);
#else
  static int rep;
  FILE *fp;

  fp = open_file("/proc/loadavg", &rep);
  if (!fp) {
    v[0] = v[1] = v[2] = 0.0;
    return;
  }

  fscanf(fp, "%f %f %f", &v[0], &v[1], &v[2]);

  fclose(fp);
#endif
}

/*
/proc/acpi/thermal_zone/THRM/cooling_mode
cooling mode:            active
/proc/acpi/thermal_zone/THRM/polling_frequency
<polling disabled>
/proc/acpi/thermal_zone/THRM/state
state:                   ok
/proc/acpi/thermal_zone/THRM/temperature
temperature:             45 C
/proc/acpi/thermal_zone/THRM/trip_points
critical (S5):           73 C
passive:                 73 C: tc1=4 tc2=3 tsp=40 devices=0xcdf6e6c0
*/

#define ACPI_THERMAL "/proc/acpi/thermal_zone/THRM/temperature"

double get_acpi_temperature() {
  static int rep;
  FILE *fp;
  double temp;

  /* TODO: should leave file open (doesn't read a whole line, so be
   * careful with buffers!) */
  fp = open_file(ACPI_THERMAL, &rep);
  if (!fp) return 0;

  fscanf(fp, "%*s %lf", &temp);
  /* fclose(fp); acpi_temp_fp = NULL; */

  return temp;
}

/*
hipo@lepakko hipo $ cat /proc/acpi/battery/BAT1/info 
present:                 yes
design capacity:         4400 mAh
last full capacity:      4064 mAh
battery technology:      rechargeable
design voltage:          14800 mV
design capacity warning: 300 mAh
design capacity low:     200 mAh
capacity granularity 1:  32 mAh
capacity granularity 2:  32 mAh
model number:            02KT
serial number:           16922
battery type:            LION
OEM info:                SANYO
*/

/*
hipo@lepakko torsmo $ cat /proc/acpi/battery/BAT1/state
present:                 yes
capacity state:          ok
charging state:          unknown
present rate:            0 mA
remaining capacity:      4064 mAh
present voltage:         16608 mV
*/

/*
2213<@jupetkellari> jupet@lagi-unstable:~$ cat /proc/apm 
2213<@jupetkellari> 1.16 1.2 0x03 0x01 0xff 0x10 -1% -1 ?
2213<@jupetkellari> (-1 ollee ei akkua kiinni, koska akku on pydll)
2214<@jupetkellari> jupet@lagi-unstable:~$ cat /proc/apm 
2214<@jupetkellari> 1.16 1.2 0x03 0x01 0x03 0x09 98% -1 ?

2238<@jupetkellari> 1.16 1.2 0x03 0x00 0x00 0x01 100% -1 ? ilman verkkovirtaa
2239<@jupetkellari> 1.16 1.2 0x03 0x01 0x00 0x01 99% -1 ? verkkovirralla

2240<@jupetkellari> 1.16 1.2 0x03 0x01 0x03 0x09 100% -1 ? verkkovirralla ja monitori pll
2241<@jupetkellari> 1.16 1.2 0x03 0x00 0x00 0x01 99% -1 ? monitori pll mutta ilman verkkovirtaa
*/

#define ACPI_BATTERY_PATH "/proc/acpi/battery/BAT1/state"
#define APM_PATH "/proc/apm"

static FILE *acpi_bat_fp;
static FILE *apm_bat_fp;

static int acpi_last_full;

static char last_battery_str[64];

static double last_battery_time;

void get_battery_stuff(char *buf, unsigned int n) {
  static int rep, rep2;

  /* don't update battery too often */
  if (current_update_time - last_battery_time < 29.5) {
    snprintf(buf, n, "%s", last_battery_str);
    return;
  }
  last_battery_time = current_update_time;

  /* first try ACPI */

  if (acpi_bat_fp == NULL && apm_bat_fp == NULL)
    acpi_bat_fp = open_file(ACPI_BATTERY_PATH, &rep);

  if (acpi_bat_fp != NULL) {
    int present_rate = -1;
    int remaining_capacity = -1;
    char charging_state[64];

    /* read last full capacity if it's zero */
    if (acpi_last_full == 0) {
      static int rep;
      FILE *fp = open_file("/proc/acpi/battery/BAT1/info", &rep);
      if (fp != NULL) {
        while (!feof(fp)) {
          char b[256];
          if (fgets(b, 256, fp) == NULL) break;

          if (sscanf(b, "last full capacity: %d", &acpi_last_full) != 0)
            break;
        }

        fclose(fp);
      }
    }

    fseek(acpi_bat_fp, 0, SEEK_SET);

    strcpy(charging_state, "unknown");

    while (!feof(acpi_bat_fp)) {
      char buf[256];
      if (fgets(buf, 256, acpi_bat_fp) == NULL) break;

      /* let's just hope units are ok */
      sscanf(buf, "charging state: %63s", charging_state);
      sscanf(buf, "present rate: %d", &present_rate);
      sscanf(buf, "remaining capacity: %d", &remaining_capacity);
    }

    if (present_rate <= 0 || remaining_capacity <= 0) {
      strcpy(last_battery_str, "AC");
    }
    else {
      if (strcmp(charging_state, "charging") == 0) {
        if (acpi_last_full != 0) {
          strcpy(last_battery_str, "charging ");
          format_time(last_battery_str+9, 63-9,
              (acpi_last_full - remaining_capacity) * 60 * 60 / present_rate);
        }
        else {
          strcpy(last_battery_str, "charging");
        }
      }
      else
        format_time(last_battery_str, 63,
            (remaining_capacity * 60 * 60) / present_rate);
    }
  }
  else {
    /* APM */
    if (apm_bat_fp == NULL)
      apm_bat_fp = open_file(APM_PATH, &rep2);

    if (apm_bat_fp != NULL) {
      int ac, status, flag, life;

      fscanf(apm_bat_fp, "%*s %*s %*x %x %x %x %d%%",
          &ac, &status, &flag, &life);

      if (life == -1) {
        /* could check now that there is ac */
        snprintf(last_battery_str, 64, "AC");
      }
      else if (ac && life != 100) { /* could check that status==3 here? */
        snprintf(last_battery_str, 64, "charging %d%%", life);
      }
      else {
        snprintf(last_battery_str, 64, "%d%%", life);
      }

      /* it seemed to buffer it so file must be closed (or could use syscalls
       * directly but I don't feel like coding it now) */
      fclose(apm_bat_fp);
      apm_bat_fp = NULL;
    }
  }

  snprintf(buf, n, "%s", last_battery_str);
}
