/*
 * Copyright (c) 2023-2025, Arm Limited. All rights reserved.
 *
 * SPDX-License-Identifier: BSD-3-Clause
 */

#include <arch.h>
#include <asm_macros.S>
#include <common/bl_common.h>
#include <cortex_a510.h>
#include <cpu_macros.S>
#include <dsu_macros.S>
#include <plat_macros.S>

/* Hardware handled coherency */
#if HW_ASSISTED_COHERENCY == 0
#error "Cortex-A510 must be compiled with HW_ASSISTED_COHERENCY enabled"
#endif

/* 64-bit only core */
#if CTX_INCLUDE_AARCH32_REGS == 1
#error "Cortex-A510 supports only AArch64. Compile with CTX_INCLUDE_AARCH32_REGS=0"
#endif

cpu_reset_prologue cortex_a510

workaround_runtime_start cortex_a510, ERRATUM(2008766), ERRATA_A510_2008766
	/* Stash ERRSELR_EL1 in x2 */
	mrs	x2, ERRSELR_EL1

	/* Select error record 0 and clear ED bit */
	msr	ERRSELR_EL1, xzr
	sysreg_bit_clear ERXCTLR_EL1, ERXCTLR_ED_BIT

	/* Select error record 1 and clear ED bit */
	mov	x0, #1
	msr	ERRSELR_EL1, x0
	sysreg_bit_clear ERXCTLR_EL1, ERXCTLR_ED_BIT

	/* Select error record 2 and clear ED bit */
	mov	x0, #2
	msr	ERRSELR_EL1, x0
	sysreg_bit_clear ERXCTLR_EL1, ERXCTLR_ED_BIT

	/* Restore ERRSELR_EL1 from x2 */
	msr	ERRSELR_EL1, x2
workaround_runtime_end cortex_a510, ERRATUM(2008766), NO_ISB

check_erratum_ls cortex_a510, ERRATUM(2008766), CPU_REV(1, 3)

workaround_reset_start cortex_a510, ERRATUM(2041909), ERRATA_A510_2041909
	/* Apply workaround */
	mov	x0, xzr
	msr	S3_6_C15_C4_0, x0
	isb

	mov	x0, #0x8500000
	msr	S3_6_C15_C4_2, x0

	mov	x0, #0x1F700000
	movk	x0, #0x8, lsl #32
	msr	S3_6_C15_C4_3, x0

	mov	x0, #0x3F1
	movk	x0, #0x110, lsl #16
	msr	S3_6_C15_C4_1, x0
workaround_reset_end cortex_a510, ERRATUM(2041909)

check_erratum_range cortex_a510, ERRATUM(2041909), CPU_REV(0, 2), CPU_REV(0, 2)

workaround_reset_start cortex_a510, ERRATUM(2042739), ERRATA_A510_2042739
	/* Apply the workaround by disabling ReadPreferUnique. */
	sysreg_bitfield_insert CORTEX_A510_CPUECTLR_EL1, CORTEX_A510_CPUECTLR_EL1_READPREFERUNIQUE_DISABLE, \
		CORTEX_A510_CPUECTLR_EL1_READPREFERUNIQUE_SHIFT, CORTEX_A510_CPUECTLR_EL1_READPREFERUNIQUE_WIDTH
workaround_reset_end cortex_a510, ERRATUM(2042739)

check_erratum_ls cortex_a510, ERRATUM(2042739), CPU_REV(0, 2)

workaround_reset_start cortex_a510, ERRATUM(2080326), ERRATA_A510_2080326
	/* Apply workaround */
	mov x0, #1
	msr S3_6_C15_C4_0, x0
	isb

	mov x0, #0x0100
	movk x0, #0x0E08, lsl #16
	msr S3_6_C15_C4_2, x0

	mov x0, #0x0300
	movk x0, #0x0F1F, lsl #16
	movk x0, #0x0008, lsl #32
	msr S3_6_C15_C4_3, x0

	mov x0, #0x03F1
	movk x0, #0x00C0, lsl #16
	msr S3_6_C15_C4_1, x0

	isb
workaround_reset_end cortex_a510, ERRATUM(2080326)

check_erratum_range cortex_a510, ERRATUM(2080326), CPU_REV(0, 2), CPU_REV(0, 2)

workaround_reset_start cortex_a510, ERRATUM(2169012), ERRATA_A510_2169012
	sysreg_bitfield_insert CORTEX_A510_CMPXACTLR_EL1, CORTEX_A510_CMPXACTLR_EL1_SNPPREFERUNIQUE_DISABLE, \
	CORTEX_A510_CMPXACTLR_EL1_SNPPREFERUNIQUE_SHIFT, CORTEX_A510_CMPXACTLR_EL1_SNPPREFERUNIQUE_WIDTH
workaround_reset_end cortex_a510, ERRATUM(2169012)

check_erratum_ls cortex_a510, ERRATUM(2169012), CPU_REV(1, 0)

workaround_reset_start cortex_a510, ERRATUM(2172148), ERRATA_A510_2172148
	/*
	 * Force L2 allocation of transient lines by setting
	 * CPUECTLR_EL1.RSCTL=0b01 and CPUECTLR_EL1.NTCTL=0b01.
	 */
	mrs	x0, CORTEX_A510_CPUECTLR_EL1
	mov	x1, #1
	bfi	x0, x1, #CORTEX_A510_CPUECTLR_EL1_RSCTL_SHIFT, #2
	bfi	x0, x1, #CORTEX_A510_CPUECTLR_EL1_NTCTL_SHIFT, #2
	msr	CORTEX_A510_CPUECTLR_EL1, x0
workaround_reset_end cortex_a510, ERRATUM(2172148)

check_erratum_ls cortex_a510, ERRATUM(2172148), CPU_REV(1, 0)

workaround_reset_start cortex_a510, ERRATUM(2218134), ERRATA_A510_2218134
	sysreg_bit_set CORTEX_A510_CPUACTLR2_EL1, BIT(43)
workaround_reset_end cortex_a510, ERRATUM(2218134)

check_erratum_range cortex_a510, ERRATUM(2218134), CPU_REV(1, 0), CPU_REV(1, 0)

workaround_reset_start cortex_a510, ERRATUM(2218950), ERRATA_A510_2218950
	/* Set bit 18 in CPUACTLR_EL1 */
	sysreg_bitfield_insert CORTEX_A510_CPUACTLR_EL1, CORTEX_A510_CPUACTLR_EL1_ALIAS_LOADSTORE_DISABLE, \
	CORTEX_A510_CPUACTLR_EL1_ALIAS_LOADSTORE_SHIFT, CORTEX_A510_CPUACTLR_EL1_ALIAS_LOADSTORE_WIDTH

	/* Set bit 25 in CMPXACTLR_EL1 */
	sysreg_bitfield_insert CORTEX_A510_CMPXACTLR_EL1, CORTEX_A510_CMPXACTLR_EL1_ALIAS_LOADSTORE_DISABLE, \
	CORTEX_A510_CMPXACTLR_EL1_ALIAS_LOADSTORE_SHIFT, CORTEX_A510_CMPXACTLR_EL1_ALIAS_LOADSTORE_WIDTH

workaround_reset_end cortex_a510, ERRATUM(2218950)

check_erratum_ls cortex_a510, ERRATUM(2218950), CPU_REV(1, 0)

workaround_reset_start cortex_a510, ERRATUM(2250311), ERRATA_A510_2250311
	/* Disable MPMM */
	mrs	x0, CPUMPMMCR_EL3
	bfm	x0, xzr, #0, #0 /* bfc instruction does not work in GCC */
	msr	CPUMPMMCR_EL3, x0
workaround_reset_end cortex_a510, ERRATUM(2250311)

check_erratum_ls cortex_a510, ERRATUM(2250311), CPU_REV(1, 0)

workaround_reset_start cortex_a510, ERRATUM(2288014), ERRATA_A510_2288014
	/* Apply the workaround by setting IMP_CPUACTLR_EL1[18] = 0b1. */
	sysreg_bitfield_insert CORTEX_A510_CPUACTLR_EL1, CORTEX_A510_CPUACTLR_EL1_DATA_CORRUPT_DISABLE, \
	CORTEX_A510_CPUACTLR_EL1_DATA_CORRUPT_SHIFT, CORTEX_A510_CPUACTLR_EL1_DATA_CORRUPT_WIDTH
workaround_reset_end cortex_a510, ERRATUM(2288014)

check_erratum_ls cortex_a510, ERRATUM(2288014), CPU_REV(1, 0)

workaround_reset_start cortex_a510, ERRATUM(2313941), ERRATA_DSU_2313941
	errata_dsu_2313941_wa_impl
workaround_reset_end cortex_a510, ERRATUM(2313941)

check_erratum_custom_start cortex_a510, ERRATUM(2313941)
	check_errata_dsu_2313941_impl
	ret
check_erratum_custom_end cortex_a510, ERRATUM(2313941)

workaround_reset_start cortex_a510, ERRATUM(2347730), ERRATA_A510_2347730
	/*
	 * Set CPUACTLR_EL1[17] to 1'b1, which disables
	 * specific microarchitectural clock gating
	 * behaviour.
	 */
	sysreg_bit_set CORTEX_A510_CPUACTLR_EL1, CORTEX_A510_CPUACTLR_EL1_BIT_17
workaround_reset_end cortex_a510, ERRATUM(2347730)

check_erratum_ls cortex_a510, ERRATUM(2347730), CPU_REV(1, 1)

workaround_reset_start cortex_a510, ERRATUM(2371937), ERRATA_A510_2371937
	/*
	 * Cacheable atomic operations can be forced
	 * to be executed near by setting
	 * IMP_CPUECTLR_EL1.ATOM=0b010. ATOM is found
	 * in [40:38] of CPUECTLR_EL1.
	 */
	sysreg_bitfield_insert CORTEX_A510_CPUECTLR_EL1, CORTEX_A510_CPUECTLR_EL1_ATOM_EXECALLINSTRNEAR, \
		CORTEX_A510_CPUECTLR_EL1_ATOM_SHIFT, CORTEX_A510_CPUECTLR_EL1_ATOM_WIDTH
workaround_reset_end cortex_a510, ERRATUM(2371937)

check_erratum_ls cortex_a510, ERRATUM(2371937), CPU_REV(1, 1)

workaround_reset_start cortex_a510, ERRATUM(2420992), ERRATA_A510_2420992
	sysreg_bit_set CORTEX_A510_CPUACTLR3_EL1, BIT(3)
workaround_reset_end cortex_a510, ERRATUM(2420992)

check_erratum_range cortex_a510, ERRATUM(2420992), CPU_REV(1, 0), CPU_REV(1, 1)

workaround_reset_start cortex_a510, ERRATUM(2666669), ERRATA_A510_2666669
	sysreg_bit_set CORTEX_A510_CPUACTLR_EL1, CORTEX_A510_CPUACTLR_EL1_BIT_38
workaround_reset_end cortex_a510, ERRATUM(2666669)

check_erratum_ls cortex_a510, ERRATUM(2666669), CPU_REV(1, 1)

.global erratum_cortex_a510_2684597_wa
workaround_runtime_start cortex_a510, ERRATUM(2684597), ERRATA_A510_2684597, CORTEX_A510_MIDR
	/*
	 * Many assemblers do not yet understand the "tsb csync" mnemonic,
	 * so use the equivalent hint instruction.
	 */
	hint	#18			/* tsb csync */
workaround_runtime_end cortex_a510, ERRATUM(2684597)

check_erratum_ls cortex_a510, ERRATUM(2684597), CPU_REV(1, 2)

.global check_erratum_cortex_a510_2971420
add_erratum_entry cortex_a510, ERRATUM(2971420), ERRATA_A510_2971420
check_erratum_range cortex_a510, ERRATUM(2971420), CPU_REV(0, 1), CPU_REV(1, 3)

workaround_reset_start cortex_a510, ERRATUM(3672349), ERRATA_A510_3672349
	/*
	 * Disable retention control for WFI and WFE by clearing both RET_CTRL
	 * fields in CPUPWRCTLR_EL1.
	 */
	sysreg_bit_clear CORTEX_A510_CPUPWRCTLR_EL1, \
			 (CORTEX_A510_CPUPWRCTLR_EL1_WFE_RET_CTRL_BITS | \
			 CORTEX_A510_CPUPWRCTLR_EL1_WFI_RET_CTRL_BITS)
workaround_reset_end cortex_a510, ERRATUM(3672349)

check_erratum_ls cortex_a510, ERRATUM(3672349), CPU_REV(1, 3)

workaround_reset_start cortex_a510, ERRATUM(3704847), ERRATA_A510_3704847
	sysreg_bit_set CORTEX_A510_CPUACTLR_EL1, BIT(9)
workaround_reset_end cortex_a510, ERRATUM(3704847)

check_erratum_ls cortex_a510, ERRATUM(3704847), CPU_REV(1, 3)

	/* ----------------------------------------------------
	 * HW will do the cache maintenance while powering down
	 * ----------------------------------------------------
	 */
func cortex_a510_core_pwr_dwn
	apply_erratum cortex_a510, ERRATUM(2008766), ERRATA_A510_2008766
	apply_erratum cortex_a510, ERRATUM(2684597), ERRATA_A510_2684597, NO_GET_CPU_REV
	/* ---------------------------------------------------
	 * Enable CPU power down bit in power control register
	 * ---------------------------------------------------
	 */
	sysreg_bit_set CORTEX_A510_CPUPWRCTLR_EL1, CORTEX_A510_CPUPWRCTLR_EL1_CORE_PWRDN_BIT
	isb
	ret
endfunc cortex_a510_core_pwr_dwn

cpu_reset_func_start cortex_a510
	/* Disable speculative loads */
	msr	SSBS, xzr
	/* skip enabling MPMM if this erratum is present */
#if ERRATA_A510_2250311
	/* the cpu_rev_var is kept in x14 */
	mov	x14, x0
	bl	check_erratum_cortex_a510_2250311
	cbz	x0, skip_mpmm
#endif
	enable_mpmm
skip_mpmm:
cpu_reset_func_end cortex_a510

	/* ---------------------------------------------
	 * This function provides Cortex-A510 specific
	 * register information for crash reporting.
	 * It needs to return with x6 pointing to
	 * a list of register names in ascii and
	 * x8 - x15 having values of registers to be
	 * reported.
	 * ---------------------------------------------
	 */
.section .rodata.cortex_a510_regs, "aS"
cortex_a510_regs:  /* The ascii list of register names to be reported */
	.asciz	"cpuectlr_el1", ""

func cortex_a510_cpu_reg_dump
	adr	x6, cortex_a510_regs
	mrs	x8, CORTEX_A510_CPUECTLR_EL1
	ret
endfunc cortex_a510_cpu_reg_dump

declare_cpu_ops cortex_a510, CORTEX_A510_MIDR, \
	cortex_a510_reset_func, \
	cortex_a510_core_pwr_dwn
