#include "stdafx.h"
#include "Nest.h"
#include "../main/Helper.h"
#include "../main/Logger.h"
#include "hardwaretypes.h"
#include "../main/RFXtrx.h"
#include "../main/SQLHelper.h"
#include "../httpclient/HTTPClient.h"
#include "../main/mainworker.h"
#include "../main/json_helper.h"

const std::string NEST_LOGIN_PATH = "https://home.nest.com/user/login";
const std::string NEST_GET_STATUS = "/v2/mobile/user.";
const std::string NEST_SET_SHARED = "/v2/put/shared.";
const std::string NEST_SET_STRUCTURE = "/v2/put/structure.";
const std::string NEST_SET_DEVICE = "/v2/put/device.";

#define NEST_USER_AGENT_STRING "User-Agent: Nest/3.0.1.15"

#ifdef _DEBUG
//#define DEBUG_NextThermostatR
//#define DEBUG_NextThermostatW
#endif

#ifdef DEBUG_NextThermostatW
void SaveString2Disk(std::string str, std::string filename)
{
	FILE *fOut = fopen(filename.c_str(), "wb+");
	if (fOut)
	{
		fwrite(str.c_str(), 1, str.size(), fOut);
		fclose(fOut);
	}
}
#endif
#ifdef DEBUG_NextThermostatR
std::string ReadFile(std::string filename)
{
	std::ifstream file;
	std::string sResult = "";
	file.open(filename.c_str());
	if (!file.is_open())
		return "";
	std::string sLine;
	while (!file.eof())
	{
		getline(file, sLine);
		sResult += sLine;
	}
	file.close();
	return sResult;
}
#endif

CNest::CNest(const int ID, const std::string &Username, const std::string &Password)
	: m_UserName(CURLEncode::URLEncode(Username))
	, m_Password(CURLEncode::URLEncode(Password))
{
	m_HwdID = ID;
	Init();
}

void CNest::Init()
{
	m_AccessToken = "";
	m_UserID = "";
	m_bDoLogin = true;
}

bool CNest::StartHardware()
{
	RequestStart();

	Init();
	// Start worker thread
	m_thread = std::make_shared<std::thread>([this] { Do_Work(); });
	SetThreadNameInt(m_thread->native_handle());
	m_bIsStarted = true;
	sOnConnected(this);
	return (m_thread != nullptr);
}

bool CNest::StopHardware()
{
	if (m_thread)
	{
		RequestStop();
		m_thread->join();
		m_thread.reset();
	}
	m_bIsStarted = false;
	return true;
}

#define NEST_POLL_INTERVAL 60

void CNest::Do_Work()
{
	Log(LOG_STATUS, "Worker started...");
	int sec_counter = NEST_POLL_INTERVAL - 5;
	while (!IsStopRequested(1000))
	{
		sec_counter++;
		if (sec_counter % 12 == 0)
		{
			m_LastHeartbeat = mytime(nullptr);
		}

		if (sec_counter % NEST_POLL_INTERVAL == 0)
		{
			GetMeterDetails();
		}
	}
	Logout();
	Log(LOG_STATUS, "Worker stopped...");
}

// Creates and updates switch used to log Heating and/or Colling.
void CNest::UpdateSwitch(const unsigned char Idx, const bool bOn, const std::string &defaultname)
{
	char szIdx[10];
	sprintf(szIdx, "%X%02X%02X%02X", 0, 0, 0, Idx);
	std::vector<std::vector<std::string>> result;
	result = m_sql.safe_query("SELECT Name,nValue,sValue FROM DeviceStatus WHERE (HardwareID==%d) AND (Type==%d) AND (SubType==%d) AND (DeviceID=='%q')", m_HwdID, pTypeLighting2, sTypeAC, szIdx);
	if (!result.empty())
	{
		// check if we have a change, if not do not update it
		int nvalue = atoi(result[0][1].c_str());
		if ((!bOn) && (nvalue == 0))
			return;
		if ((bOn && (nvalue != 0)))
			return;
	}

	// Send as Lighting 2
	tRBUF lcmd;
	memset(&lcmd, 0, sizeof(RBUF));
	lcmd.LIGHTING2.packetlength = sizeof(lcmd.LIGHTING2) - 1;
	lcmd.LIGHTING2.packettype = pTypeLighting2;
	lcmd.LIGHTING2.subtype = sTypeAC;
	lcmd.LIGHTING2.id1 = 0;
	lcmd.LIGHTING2.id2 = 0;
	lcmd.LIGHTING2.id3 = 0;
	lcmd.LIGHTING2.id4 = Idx;
	lcmd.LIGHTING2.unitcode = 1;
	int level = 15;
	if (!bOn)
	{
		level = 0;
		lcmd.LIGHTING2.cmnd = light2_sOff;
	}
	else
	{
		level = 15;
		lcmd.LIGHTING2.cmnd = light2_sOn;
	}
	lcmd.LIGHTING2.level = (BYTE)level;
	lcmd.LIGHTING2.filler = 0;
	lcmd.LIGHTING2.rssi = 12;
	sDecodeRXMessage(this, (const unsigned char *)&lcmd.LIGHTING2, defaultname.c_str(), 255, m_Name.c_str());
}

bool CNest::Login()
{
	if (!m_AccessToken.empty())
	{
		Logout();
	}
	m_AccessToken = "";
	m_UserID = "";

	std::stringstream sstr;
	sstr << "username=" << m_UserName << "&password=" << m_Password;
	std::string szPostdata = sstr.str();
	std::vector<std::string> ExtraHeaders;
	ExtraHeaders.push_back(NEST_USER_AGENT_STRING);
	std::string sResult;

	std::string sURL = NEST_LOGIN_PATH;
	if (!HTTPClient::POST(sURL, szPostdata, ExtraHeaders, sResult))
	{
		Log(LOG_ERROR, "Error login!");
		return false;
	}

	Json::Value root;
	bool bRet = ParseJSon(sResult, root);
	if ((!bRet) || (!root.isObject()))
	{
		Log(LOG_ERROR, "Invalid data received, or invalid username/password!");
		return false;
	}
	if (root["urls"].empty())
	{
		Log(LOG_ERROR, "Invalid data received, or invalid username/password!");
		return false;
	}
	if (root["urls"]["transport_url"].empty())
	{
		Log(LOG_ERROR, "Invalid data received, or invalid username/password!");
		return false;
	}
	m_TransportURL = root["urls"]["transport_url"].asString();

	if (root["access_token"].empty())
	{
		Log(LOG_ERROR, "Invalid data received, or invalid username/password!");
		return false;
	}
	m_AccessToken = root["access_token"].asString();

	if (root["userid"].empty())
	{
		Log(LOG_ERROR, "Invalid data received, or invalid username/password!");
		return false;
	}
	m_UserID = root["userid"].asString();

	m_bDoLogin = false;
	return true;
}

void CNest::Logout()
{
	if (m_bDoLogin)
		return; // we are not logged in
	m_AccessToken = "";
	m_UserID = "";
	m_bDoLogin = true;
}

bool CNest::WriteToHardware(const char *pdata, const unsigned char /*length*/)
{
	if (m_UserName.empty())
		return false;
	if (m_Password.empty())
		return false;

	const tRBUF *pCmd = reinterpret_cast<const tRBUF *>(pdata);
	if (pCmd->LIGHTING2.packettype != pTypeLighting2)
		return false; // later add RGB support, if someone can provide access

	int node_id = pCmd->LIGHTING2.id4;

	bool bIsOn = (pCmd->LIGHTING2.cmnd == light2_sOn);

	if (node_id % 3 == 0)
	{
		// Away
		return SetAway((uint8_t)node_id, bIsOn);
	}

	if (node_id % 4 == 0)
	{
		// Manual Eco Mode
		return SetManualEcoMode((uint8_t)node_id, bIsOn);
	}

	return false;
}

void CNest::UpdateSmokeSensor(const unsigned char Idx, const bool bOn, const std::string &defaultname)
{
	bool bDeviceExits = true;
	char szIdx[10];
	sprintf(szIdx, "%X%02X%02X%02X", 0, 0, Idx, 0);
	std::vector<std::vector<std::string>> result;
	result = m_sql.safe_query("SELECT Name,nValue,sValue FROM DeviceStatus WHERE (HardwareID==%d) AND (DeviceID=='%q')", m_HwdID, szIdx);
	if (result.empty())
	{
		bDeviceExits = false;
	}
	else
	{
		// check if we have a change, if not only update the LastUpdate field
		bool bNoChange = false;
		int nvalue = atoi(result[0][1].c_str());
		if ((!bOn) && (nvalue == 0))
			bNoChange = true;
		else if ((bOn && (nvalue != 0)))
			bNoChange = true;
		if (bNoChange)
		{
			std::string sLastUpdate = TimeToString(nullptr, TF_DateTime);
			m_sql.safe_query("UPDATE DeviceStatus SET LastUpdate='%q' WHERE (HardwareID == %d) AND (DeviceID == '%q')", sLastUpdate.c_str(), m_HwdID, szIdx);
			return;
		}
	}

	// Send as Lighting 2
	tRBUF lcmd;
	memset(&lcmd, 0, sizeof(RBUF));
	lcmd.LIGHTING2.packetlength = sizeof(lcmd.LIGHTING2) - 1;
	lcmd.LIGHTING2.packettype = pTypeLighting2;
	lcmd.LIGHTING2.subtype = sTypeAC;
	lcmd.LIGHTING2.id1 = 0;
	lcmd.LIGHTING2.id2 = 0;
	lcmd.LIGHTING2.id3 = Idx;
	lcmd.LIGHTING2.id4 = 0;
	lcmd.LIGHTING2.unitcode = 1;
	int level = 15;
	if (!bOn)
	{
		level = 0;
		lcmd.LIGHTING2.cmnd = light2_sOff;
	}
	else
	{
		level = 15;
		lcmd.LIGHTING2.cmnd = light2_sOn;
	}
	lcmd.LIGHTING2.level = (uint8_t)level;
	lcmd.LIGHTING2.filler = 0;
	lcmd.LIGHTING2.rssi = 12;

	if (!bDeviceExits)
	{
		m_mainworker.PushAndWaitRxMessage(this, (const unsigned char *)&lcmd.LIGHTING2, defaultname.c_str(), 255, m_Name.c_str());
		// Assign default name for device
		m_sql.safe_query("UPDATE DeviceStatus SET Name='%q' WHERE (HardwareID==%d) AND (DeviceID=='%q')", defaultname.c_str(), m_HwdID, szIdx);
		result = m_sql.safe_query("SELECT ID FROM DeviceStatus WHERE (HardwareID==%d) AND (DeviceID=='%q')", m_HwdID, szIdx);
		if (!result.empty())
		{
			m_sql.safe_query("UPDATE DeviceStatus SET SwitchType=%d WHERE (ID=='%q')", STYPE_SMOKEDETECTOR, result[0][0].c_str());
		}
	}
	else
		sDecodeRXMessage(this, (const unsigned char *)&lcmd.LIGHTING2, defaultname.c_str(), 255, m_Name.c_str());
}

void CNest::GetMeterDetails()
{
	std::string sResult;
#ifdef DEBUG_NextThermostatR
	sResult = ReadFile("E:\\nest.json");
#else
	if (m_UserName.empty())
		return;
	if (m_Password.empty())
		return;
	if (m_bDoLogin)
	{
		if (!Login())
			return;
	}
	std::vector<std::string> ExtraHeaders;

	ExtraHeaders.push_back(NEST_USER_AGENT_STRING);
	ExtraHeaders.push_back("Authorization:Basic " + m_AccessToken);
	ExtraHeaders.push_back("X-nl-user-id:" + m_UserID);
	ExtraHeaders.push_back("X-nl-protocol-version:1");

	// Get Data
	std::string sURL = m_TransportURL + NEST_GET_STATUS + m_UserID;
	if (!HTTPClient::GET(sURL, ExtraHeaders, sResult))
	{
		Log(LOG_ERROR, "Error getting current state!");
		m_bDoLogin = true;
		return;
	}
#endif

#ifdef DEBUG_NextThermostatW
	SaveString2Disk(sResult, "E:\\nest.json");
#endif

	Json::Value root;
	bool bRet = ParseJSon(sResult, root);
	if ((!bRet) || (!root.isObject()))
	{
		Log(LOG_ERROR, "Invalid data received!");
		m_bDoLogin = true;
		return;
	}
	bool bHaveShared = !root["shared"].empty();
	bool bHaveTopaz = !root["topaz"].empty();

	if ((!bHaveShared) && (!bHaveTopaz))
	{
		Log(LOG_ERROR, "request not successful, restarting..!");
		m_bDoLogin = true;
		return;
	}

	// Protect
	if (bHaveTopaz)
	{
		if (root["topaz"].empty())
		{
			Log(LOG_ERROR, "request not successful, restarting..!");
			m_bDoLogin = true;
			return;
		}
		Json::Value::Members members = root["topaz"].getMemberNames();
		if (members.empty())
		{
			Log(LOG_ERROR, "request not successful, restarting..!");
			m_bDoLogin = true;
			return;
		}
		int SwitchIndex = 1;
		for (auto itDevice = root["topaz"].begin(); itDevice != root["topaz"].end(); ++itDevice)
		{
			Json::Value device = *itDevice;
			std::string devstring = itDevice.key().asString();
			if (device["where_id"].empty())
				continue;
			std::string whereid = device["where_id"].asString();
			// lookup name
			std::string devName = devstring;
			if (!root["where"].empty())
			{
				for (auto iwhere : root["where"])
				{
					if (!iwhere["wheres"].empty())
					{
						for (auto iwhereItt : iwhere["wheres"])
						{
							if (!iwhereItt["where_id"].empty())
							{
								std::string tmpWhereid = iwhereItt["where_id"].asString();
								if (tmpWhereid == whereid)
								{
									devName = iwhereItt["name"].asString();
									break;
								}
							}
						}
					}
				}
			}
			bool bIAlarm = false;
			bool bBool;
			if (!device["component_speaker_test_passed"].empty())
			{
				bBool = device["component_speaker_test_passed"].asBool();
				if (!bBool)
					bIAlarm = true;
			}
			if (!device["component_smoke_test_passed"].empty())
			{
				bBool = device["component_smoke_test_passed"].asBool();
				if (!bBool)
					bIAlarm = true;
			}
			/*
						if (!device["component_heat_test_passed"].empty())
						{
							bBool = device["component_heat_test_passed"].asBool();
							if (!bBool)
								bIAlarm = true;
						}
			*/
			if (!device["component_buzzer_test_passed"].empty())
			{
				bBool = device["component_buzzer_test_passed"].asBool();
				if (!bBool)
					bIAlarm = true;
			}
			/*
						if (!device["component_us_test_passed"].empty())
						{
							bBool = device["component_us_test_passed"].asBool();
							if (!bBool)
								bIAlarm = true;
						}
			*/
			if (!device["component_temp_test_passed"].empty())
			{
				bBool = device["component_temp_test_passed"].asBool();
				if (!bBool)
					bIAlarm = true;
			}
			if (!device["component_wifi_test_passed"].empty())
			{
				bBool = device["component_wifi_test_passed"].asBool();
				if (!bBool)
					bIAlarm = true;
			}
			if (!device["component_als_test_passed"].empty())
			{
				bBool = device["component_als_test_passed"].asBool();
				if (!bBool)
					bIAlarm = true;
			}
			if (!device["component_co_test_passed"].empty())
			{
				bBool = device["component_co_test_passed"].asBool();
				if (!bBool)
					bIAlarm = true;
			}
			if (!device["component_hum_test_passed"].empty())
			{
				bBool = device["component_hum_test_passed"].asBool();
				if (!bBool)
					bIAlarm = true;
			}
			UpdateSmokeSensor((uint8_t)SwitchIndex, bIAlarm, devName);
			SwitchIndex++;
		}
	}

	// Thermostat
	if (!bHaveShared)
		return;
	if (root["shared"].empty())
	{
		if (bHaveTopaz)
			return;
		Log(LOG_ERROR, "request not successful, restarting..!");
		m_bDoLogin = true;
		return;
	}

	int iThermostat = 0;
	for (auto ittStructure = root["structure"].begin(); ittStructure != root["structure"].end(); ++ittStructure)
	{
		Json::Value nstructure = *ittStructure;
		if (!nstructure.isObject())
			continue;
		std::string StructureID = ittStructure.key().asString();
		std::string StructureName = nstructure["name"].asString();

		for (auto &ittDevice : nstructure["devices"])
		{
			std::string devID = ittDevice.asString();
			if (devID.find("device.") == std::string::npos)
				continue;
			std::string Serial = devID.substr(7);
			if (root["device"].empty())
				continue;
			if (root["device"][Serial].empty())
				continue; // not found !?
			if (root["shared"][Serial].empty())
				continue; // Nothing shared?

			Json::Value ndevice = root["device"][Serial];
			if (!ndevice.isObject())
				continue;

			std::string Name = "Thermostat";
			if (!ndevice["where_id"].empty())
			{
				// Lookup our 'where' (for the Name of the thermostat)
				std::string where_id = ndevice["where_id"].asString();

				if (!root["where"].empty())
				{
					if (!root["where"][StructureID].empty())
					{
						for (auto nwheres : root["where"][StructureID]["wheres"])
						{
							if (nwheres["where_id"] == where_id)
							{
								Name = StructureName + " " + nwheres["name"].asString();
								break;
							}
						}
					}
				}
			}

			_tNestThemostat ntherm;
			ntherm.Serial = Serial;
			ntherm.StructureID = StructureID;
			ntherm.Name = Name;
			m_thermostats[iThermostat] = ntherm;

			Json::Value nshared = root["shared"][Serial];

			// Setpoint
			if (!nshared["target_temperature"].empty())
			{
				float currentSetpoint = nshared["target_temperature"].asFloat();
				SendSetPointSensor(0, 0, 0, (const unsigned char)(iThermostat * 3) + 1, 0, 255, currentSetpoint, Name + " Setpoint");
			}
			// Room Temperature/Humidity
			if (!nshared["current_temperature"].empty())
			{
				float currentTemp = nshared["current_temperature"].asFloat();
				int Humidity = root["device"][Serial]["current_humidity"].asInt();
				SendTempHumSensor((iThermostat * 3) + 2, 255, currentTemp, Humidity, Name + " TempHum");
			}

			// Check if thermostat is currently Heating
			if (nshared["can_heat"].asBool() && !nshared["hvac_heater_state"].empty())
			{
				bool bIsHeating = nshared["hvac_heater_state"].asBool();
				UpdateSwitch((unsigned char)(113 + (iThermostat * 3)), bIsHeating, Name + " HeatingOn");
			}

			// Check if thermostat is currently Cooling
			if (nshared["can_cool"].asBool() && !nshared["hvac_ac_state"].empty())
			{
				bool bIsCooling = nshared["hvac_ac_state"].asBool();
				UpdateSwitch((unsigned char)(114 + (iThermostat * 3)), bIsCooling, Name + " CoolingOn");
			}

			// Away
			if (!nstructure["away"].empty())
			{
				bool bIsAway = nstructure["away"].asBool();
				SendSwitch((iThermostat * 3) + 3, 1, 255, bIsAway, 0, Name + " Away", m_Name);
			}

			// Manual Eco mode
			if (!ndevice["eco"]["mode"].empty())
			{
				std::string sCurrentHvacMode = ndevice["eco"]["mode"].asString();
				bool bIsManualEcoMode = (sCurrentHvacMode == "manual-eco");
				SendSwitch((iThermostat * 3) + 4, 1, 255, bIsManualEcoMode, 0, Name + " Manual Eco Mode", m_Name);
			}

			iThermostat++;
		}
	}
}

void CNest::SetSetpoint(const int idx, const float temp)
{
	if (m_UserName.empty())
		return;
	if (m_Password.empty())
		return;

	if (m_bDoLogin == true)
	{
		if (!Login())
			return;
	}
	int iThermostat = (idx - 1) / 3;
	if (iThermostat > (int)m_thermostats.size())
		return;

	std::vector<std::string> ExtraHeaders;

	ExtraHeaders.push_back(NEST_USER_AGENT_STRING);
	ExtraHeaders.push_back("Authorization:Basic " + m_AccessToken);
	ExtraHeaders.push_back("X-nl-protocol-version:1");

	float tempDest = temp;
	unsigned char tSign = m_sql.m_tempsign[0];
	if (tSign == 'F')
	{
		tempDest = static_cast<float>(ConvertToCelsius(tempDest));
	}

	Json::Value root;
	root["target_change_pending"] = true;
	root["target_temperature"] = tempDest;

	std::string sResult;

	std::string sURL = m_TransportURL + NEST_SET_SHARED + m_thermostats[iThermostat].Serial;
	if (!HTTPClient::POST(sURL, root.toStyledString(), ExtraHeaders, sResult, true, true))
	{
		Log(LOG_ERROR, "Error setting setpoint!");
		m_bDoLogin = true;
		return;
	}
	GetMeterDetails();
}

bool CNest::SetAway(const unsigned char Idx, const bool bIsAway)
{
	if (m_UserName.empty())
		return false;
	if (m_Password.empty())
		return false;

	if (m_bDoLogin == true)
	{
		if (!Login())
			return false;
	}

	int iThermostat = (Idx - 3) / 3;
	if (iThermostat > (int)m_thermostats.size())
		return false;

	std::vector<std::string> ExtraHeaders;

	ExtraHeaders.push_back(NEST_USER_AGENT_STRING);
	ExtraHeaders.push_back("Authorization:Basic " + m_AccessToken);
	ExtraHeaders.push_back("X-nl-protocol-version:1");

	Json::Value root;
	root["away"] = bIsAway;
	root["away_timestamp"] = (int)mytime(nullptr);
	root["away_setter"] = 0;

	std::string sResult;

	std::string sURL = m_TransportURL + NEST_SET_STRUCTURE + m_thermostats[iThermostat].StructureID;
	if (!HTTPClient::POST(sURL, root.toStyledString(), ExtraHeaders, sResult, true, true))
	{
		Log(LOG_ERROR, "Error setting away mode!");
		m_bDoLogin = true;
		return false;
	}
	return true;
}

bool CNest::SetManualEcoMode(const unsigned char Idx, const bool bIsManualEcoMode)
{
	if (m_UserName.empty())
		return false;
	if (m_Password.empty())
		return false;

	if (m_bDoLogin == true)
	{
		if (!Login())
			return false;
	}

	int iThermostat = (Idx - 4) / 3;
	if (iThermostat > (int)m_thermostats.size())
		return false;

	std::vector<std::string> ExtraHeaders;

	ExtraHeaders.push_back(NEST_USER_AGENT_STRING);
	ExtraHeaders.push_back("Authorization:Basic " + m_AccessToken);
	ExtraHeaders.push_back("X-nl-protocol-version:1");

	Json::Value root;
	Json::Value eco;

	eco["mode"] = (bIsManualEcoMode ? "manual-eco" : "schedule");
	root["eco"] = eco;

	std::string sResult;

	// If thermostat information has not yet been read we can't do anything so let's fail.
	if (m_thermostats[iThermostat].Serial.empty())
		return false;

	std::string sURL = m_TransportURL + NEST_SET_DEVICE + m_thermostats[iThermostat].Serial;
	if (!HTTPClient::POST(sURL, root.toStyledString(), ExtraHeaders, sResult, true, true))
	{
		Log(LOG_ERROR, "Error setting manual eco mode!");
		m_bDoLogin = true;
		return false;
	}
	return true;
}

void CNest::SetProgramState(const int /*newState*/)
{
	if (m_UserName.empty())
		return;
	if (m_Password.empty())
		return;

	if (m_bDoLogin)
	{
		if (!Login())
			return;
	}
}
