// SPDX-License-Identifier: GPL-2.0
/*
 * Copyright (C) 2024 Marek Behún <kabel@kernel.org>
 */
#include <errno.h>
#include <stdarg.h>
#include <stdbool.h>
#include <stdio.h>
#include <stdlib.h>
#include <stdint.h>
#include <string.h>
#include <strings.h>
#include <endian.h>
#include <byteswap.h>
#include <fcntl.h>
#include <unistd.h>

#include "crc32.h"

#define OMNIA_EEPROM_MAGIC		0x0341a034
#define OMNIA_EEPROM_SYSFS_PATH		"/sys/devices/platform/soc/soc:internal-regs/f1011000.i2c/i2c-0/i2c-1/1-0054/eeprom"
#define OMNIA_EEPROM_SYSFS_PATH_ALT	"/sys/bus/i2c/devices/1-0054/eeprom"

struct omnia_eeprom {
	uint32_t magic;
	uint32_t ramsize;
	char region[4];
	uint32_t crc;

	/* second part (only considered if crc2 is not all-ones) */
	char ddr_speed[5];
	uint8_t old_ddr_training;
	uint8_t reserved[38];
	uint32_t crc2;
};

static const char *argv0;

__attribute__((__noreturn__, __format__(__printf__, 1, 2)))
static void die(const char *fmt, ...)
{
	int saved_errno;
	va_list ap;

	saved_errno = errno;

	fflush(stdout);
	fflush(stderr);

	while (*fmt == '\n') {
		fputc('\n', stderr);
		++fmt;
	}

	fprintf(stderr, "%s: ", argv0);

	errno = saved_errno;

	va_start(ap, fmt);
	vfprintf(stderr, fmt, ap);
	va_end(ap);

	fputc('\n', stderr);

	exit(EXIT_FAILURE);
}

static void omnia_eeprom_to_host(struct omnia_eeprom *oep)
{
	oep->magic = le32toh(oep->magic);
	oep->ramsize = le32toh(oep->ramsize);
	oep->crc = le32toh(oep->crc);
	oep->crc2 = le32toh(oep->crc2);
}

static void omnia_eeprom_to_le32(struct omnia_eeprom *oep)
{
	oep->magic = htole32(oep->magic);
	oep->ramsize = htole32(oep->ramsize);
	oep->crc = htole32(oep->crc);
	oep->crc2 = htole32(oep->crc2);
}

static int open_eeprom(int flags)
{
	int fd;

	fd = open(OMNIA_EEPROM_SYSFS_PATH, flags);
	if (fd >= 0 || errno != ENOENT)
		return fd;

	return open(OMNIA_EEPROM_SYSFS_PATH_ALT, flags);
}

static void omnia_eeprom_read(struct omnia_eeprom *oep)
{
	ssize_t rd;
	int fd;

	fd = open_eeprom(O_RDONLY);
	if (fd < 0)
		die("could not open EEPROM for reading: %m");

	rd = read(fd, oep, sizeof(*oep));
	if (rd < 0)
		die("could not read EEPROM: %m");
	else if (rd < sizeof(*oep))
		die("could not read whole Turris Omnia EEPROM structure");

	close(fd);

	omnia_eeprom_to_host(oep);
}

static void omnia_eeprom_write(const struct omnia_eeprom *_oep)
{
	struct omnia_eeprom oep = *_oep;
	ssize_t wr;
	int fd;

	omnia_eeprom_to_le32(&oep);

	fd = open_eeprom(O_WRONLY);
	if (fd < 0)
		die("could not open EEPROM for writing: %m");

	wr = write(fd, &oep, sizeof(oep));
	if (wr < 0)
		die("could not write EEPROM: %m");
	else if (wr < sizeof(oep))
		die("could not write whole Turris Omnia EEPROM structure");

	close(fd);
}

static bool is_all_ones(const void *buf, size_t size)
{
	const uint8_t *p = buf;

	while (size--)
		if (*p++ != 0xff)
			return false;

	return true;
}

static uint32_t omnia_eeprom_crc(const struct omnia_eeprom *oep)
{
	return ~crc32(~0, oep, 12);
}

static uint32_t omnia_eeprom_crc2(const struct omnia_eeprom *oep)
{
	return ~crc32(~0, oep, 60);
}

static void _omnia_eeprom_print(const struct omnia_eeprom *oep)
{
	char inv_magic[32] = "", inv_crc[32] = "", inv_crc2[32] = "";
	char old_ddr_training[] = "(empty, defaults to 0)";
	char ddr_speed[] = "(empty, defaults to 1600K)";
	uint32_t exp_crc, exp_crc2;

	if (oep->ddr_speed[0] != '\0' && oep->ddr_speed[0] != 0xff)
		sprintf(ddr_speed, "%.5s", oep->ddr_speed);

	if (oep->old_ddr_training != 0xff)
		sprintf(old_ddr_training, "%u", oep->old_ddr_training);

	if (oep->magic != OMNIA_EEPROM_MAGIC)
		sprintf(inv_magic, "   (INVALID, expected %08x)", __builtin_bswap32(OMNIA_EEPROM_MAGIC));

	exp_crc = omnia_eeprom_crc(oep);
	if (oep->crc != exp_crc)
		sprintf(inv_crc, "   (INVALID, expected %08x)", __builtin_bswap32(exp_crc));

	exp_crc2 = omnia_eeprom_crc2(oep);
	if (oep->crc2 != exp_crc2 && !is_all_ones(&oep->ddr_speed, 48))
		sprintf(inv_crc2, "   (INVALID, expected %08x)", __builtin_bswap32(exp_crc2));

	printf("Magic constant                %08x%s\n", __builtin_bswap32(oep->magic), inv_magic);
	printf("RAM size in GB                %u\n", oep->ramsize);
	printf("Wi-Fi Region                  %.4s\n", oep->region);
	printf("CRC32 checksum                %08x%s\n", __builtin_bswap32(oep->crc), inv_crc);
	printf("DDR speed                     %s\n", ddr_speed);
	printf("Use old DDR training          %s\n", old_ddr_training);
	printf("Reserved fields               (%u bytes)\n", (unsigned int)sizeof(oep->reserved));
	printf("Extended CRC32 checksum       %08x%s\n", __builtin_bswap32(oep->crc2), inv_crc2);
}

static void omnia_eeprom_print(void)
{
	struct omnia_eeprom oep;

	omnia_eeprom_read(&oep);
	_omnia_eeprom_print(&oep);
}

static void omnia_eeprom_get(const char *field, char value[static 6])
{
	struct omnia_eeprom oep;

	omnia_eeprom_read(&oep);

	if (oep.magic != OMNIA_EEPROM_MAGIC)
		die("Invalid magic value %08x in EEPROM", __builtin_bswap32(oep.magic));

	if (oep.crc != omnia_eeprom_crc(&oep))
		die("Invalid first CRC %08x in EEPROM", __builtin_bswap32(oep.crc));

	/* check second CRC only if requested field is from that part of the structure */
	if ((!strcmp(field, "ddr_speed") || !strcmp(field, "old_ddr_training")) &&
	    (oep.crc2 != omnia_eeprom_crc2(&oep) && !is_all_ones(&oep.ddr_speed, 48)))
		die("Invalid second CRC %08x in EEPROM", __builtin_bswap32(oep.crc2));

	if (!strcmp(field, "ram_size")) {
		sprintf(value, "%u", oep.ramsize);
	} else if (!strcmp(field, "wifi_region")) {
		if (oep.region[0] == '\0' || oep.region[0] == 0xff) {
			value[0] = '\0';
		} else {
			sprintf(value, "%.4s", oep.region);
			if (value[2] == ' ')
				value[2] = '\0';
		}
	} else if (!strcmp(field, "ddr_speed")) {
		if (oep.ddr_speed[0] == '\0' || oep.ddr_speed[0] == 0xff)
			value[0] = '\0';
		else
			strlcpy(value, oep.ddr_speed, 6);
	} else if (!strcmp(field, "old_ddr_training")) {
		if (oep.old_ddr_training == 0xff)
			value[0] = '\0';
		else
			sprintf(value, "%u", oep.old_ddr_training);
	} else {
		die("unknown EEPROM field '%s'", field);
	}
}

static void omnia_eeprom_set_ram_size(struct omnia_eeprom *oep, const char *value)
{
	if (!value || value[0] == '\0')
		die("'ram_size' field cannot be empty");

	if ((value[0] != '1' && value[0] != '2') || value[1] != '\0')
		die("invalid value '%s' for the 'ram_size' field. Valid values are: 1, 2", value);

	oep->ramsize = value[0] - '0';
}

static void omnia_eeprom_set_wifi_region(struct omnia_eeprom *oep, const char *value)
{
	if (!value || value[0] == '\0')
		die("'wifi_region' field cannot be empty");

	if (strlen(value) != 2)
		die("'wifi_region' field has to be 2 characters long");

	memcpy(oep->region, value, 2);
	oep->region[2] = '\0';
}

static void omnia_eeprom_set_ddr_speed(struct omnia_eeprom *oep, const char *value)
{
	if (!value || value[0] == '\0')
		memset(oep->ddr_speed, 0xff, sizeof(oep->ddr_speed));
	else if (!strcmp(value, "1333H") || !strcmp(value, "1600K"))
		memcpy(oep->ddr_speed, value, sizeof(oep->ddr_speed));
	else
		die("invalid value '%s' for the 'ddr_speed' field. Valid values are: 1333H, 1600K", value);
}

static void omnia_eeprom_set_old_ddr_training(struct omnia_eeprom *oep, const char *value)
{
	if (!value || value[0] == '\0')
		oep->old_ddr_training = 0xff;
	else if (!strcmp(value, "1") || !strcmp(value, "yes") || !strcmp(value, "true"))
		oep->old_ddr_training = 1;
	else if (!strcmp(value, "0") || !strcmp(value, "no") || !strcmp(value, "false"))
		oep->old_ddr_training = 0;
	else
		die("invalid value '%s' for the 'old_ddr_training' field. Valid values are: 1/yes/true, 0/no/false",
		    value);
}

static void omnia_eeprom_set(const char *field, const char *value)
{
	struct omnia_eeprom oep;

	omnia_eeprom_read(&oep);

	oep.magic = OMNIA_EEPROM_MAGIC;

	if (!strcmp(field, "ram_size"))
		omnia_eeprom_set_ram_size(&oep, value);
	else if (!strcmp(field, "wifi_region"))
		omnia_eeprom_set_wifi_region(&oep, value);
	else if (!strcmp(field, "ddr_speed"))
		omnia_eeprom_set_ddr_speed(&oep, value);
	else if (!strcmp(field, "old_ddr_training"))
		omnia_eeprom_set_old_ddr_training(&oep, value);
	else
		die("unknown EEPROM field '%s'", field);

	oep.crc = omnia_eeprom_crc(&oep);
	oep.crc2 = omnia_eeprom_crc2(&oep);

	omnia_eeprom_write(&oep);
}

static void usage(void)
{
	printf("Usage: omnia-eeprom <subcommand> [<args>]\n\n"
		"Subcommands:\n"
		"  print                print EEPROM fields (default option)\n"
		"  get <field>          print value of field\n"
		"  set <field> [value]  set field's value\n"
		"  unset <field>        unset field\n\n"
		"Available fields:\n"
		"  ram_size             RAM size in GB (either 1 or 2)\n"
		"  wifi_region          Wi-Fi region\n"
		"  ddr_speed            DDR speed (1333H or 1600K)\n"
		"  old_ddr_training     use old DDR training algorithm\n\n"
		"Other options:\n"
		"  -h, --help           show this help\n"
		"  -v, --version        show omnia-eeprom's version\n\n");
}

int main(int argc, char **argv)
{
	argv0 = argv[0];

	if (argc < 2 || !strcmp(argv[1], "print")) {
		omnia_eeprom_print();
	} else if (!strcmp(argv[1], "get")) {
		char value[6];

		if (argc != 3)
			die("the 'get' subcommand requires exatly one argument");

		omnia_eeprom_get(argv[2], value);
		printf("%s\n", value);
	} else if (!strcmp(argv[1], "set")) {
		if (argc != 4)
			die("the 'set' subcommand requires exactly two arguments");

		omnia_eeprom_set(argv[2], argv[3]);
	} else if (!strcmp(argv[1], "unset")) {
		if (argc != 3)
			die("the 'unset' subcommand requires exatly one argument");

		omnia_eeprom_set(argv[2], NULL);
	} else if (!strcmp(argv[1], "-h") || !strcmp(argv[1], "--help")) {
		usage();
	} else if (!strcmp(argv[1], "-v") || !strcmp(argv[1], "--version")) {
		printf("omnia-eeprom " OMNIA_EEPROM_VERSION " (built on " __DATE__ " " __TIME__ ")\n"
			"Copyright (C) 2024 Marek Behun\n"
			"License: GNU GPL version 2.\n");
	} else {
		die("unrecognized %s '%s'", argv[1][0] == '-' ? "option" : "subcommand", argv[1]);
	}

	exit(EXIT_SUCCESS);
}
